<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector2BA\Http\TwoBA;
use Daylight\Connector2BA\Jobs\ProcessSelectionProfile;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use ZipArchive;

/**
 * Command to download and extract SelectionProfile files from 2BA API
 * The files are downloaded from the provided CDN link in the API response
 * and extracted to storage/app/2ba_selection_profiles/
 * Optionally, a processing job can be queued after extraction using the --process flag
 *
 * The table 2ba_data is used to store metadata about the downloaded products.
 * Do this one a weekly or monthly, as the data does not change often.
 *
 * @examples
 * - php artisan connector:download-selection-profile-2ba --directory="2ba_selection_profiles/2ba_selection_profile_b8de0b5a-54ca-4673-bfa1-531b135d63e7_2025-10-01_12-10-04" --process
 * - php artisan connector:download-selection-profile-2ba --directory="2ba_selection_profiles/2ba_selection_profile_dc750459-bf8d-4f8d-a8fc-a7a27f08e0d4_2025-10-24_14-38-19" --process-now --supplier
 * - php artisan connector:download-selection-profile-2ba --directory="2ba_selection_profiles/2ba_selection_profile_dc750459-bf8d-4f8d-a8fc-a7a27f08e0d4_2025-11-02_20-28-17" --process-now
 * - php artisan connector:download-selection-profile-2ba --process-now --selectionprofile=df8ad5cc-3752-410b-a3bd-42f078b57ed6
 */

class DownloadSelectionProfile extends Command
{
    protected $signature = 'connector:download-selection-profile-2ba
        {--process : Queue processing job after extraction}
        {--process-now : Process immediately without queue}
        {--directory= : Process existing directory instead of downloading new profile}
        {--supplier : Optional to trigger filter on owner products, when loading the assortment from the owner out of 2BA}
        {--selectionprofile= : Optional SelectionProfile ID to override config value}';

    protected $description = 'Download and extract SelectionProfile files from 2BA API or process existing directory';

    private TwoBA $connector;

    public function handle(): int
    {
        // Check if a directory option is provided
        $existingDirectory = $this->option('directory');
        $supplierIdGln = $this->option('supplier') ? config('services.connector.2ba.supplier_id_gln') : null;

        if ($existingDirectory) {
            return $this->processExistingDirectory($existingDirectory, $supplierIdGln);
        }

        // Original download flow
        $this->connector = new TwoBA();
        if ($this->option('selectionprofile')) {
            $selectionProfileId = $this->option('selectionprofile');
        } else {
            $selectionProfileId = config('services.connector.2ba.selection_profile_id');
        }

        if (!$selectionProfileId) {
            $this->error('CONNECTOR_2BA_SELECTION_PROFILE_ID not configured');
            return Command::FAILURE;
        }

        $this->line("Starting SelectionProfile download for ID: {$selectionProfileId}");

        try {
            // Step 1: Get download information from 2BA API
            $downloadInfo = $this->getDownloadInfo($selectionProfileId);

            if (!$downloadInfo) {
                $this->error('Failed to get download information from 2BA API');
                return Command::FAILURE;
            }

            // Step 2: Download the file from the CDN link
            $tempFilePath = $this->downloadFileFromCdn($downloadInfo);

            if (!$tempFilePath) {
                $this->error('Failed to download SelectionProfile file');
                return Command::FAILURE;
            }

            $this->info("File downloaded successfully: {$tempFilePath}");

            // Step 3: Extract the ZIP file
            $extractedPath = $this->extractZipFile($tempFilePath);

            if (!$extractedPath) {
                $this->error('Failed to extract ZIP file');
                return Command::FAILURE;
            }

            $this->info("File extracted successfully to: {$extractedPath}");

            // Step 4: Clean up temporary download file
            Storage::delete($tempFilePath);
            $this->line("Temporary download file cleaned up");

            // Step 5: Process files based on options
            if ($this->option('process-now')) {
                $this->line("Processing files immediately (no queue)...");
                $this->processDirectoryNow($selectionProfileId, $extractedPath, $supplierIdGln);
                $this->info("File processing completed!");
            } elseif ($this->option('process')) {
                $this->line("Dispatching processing job for extracted files...");
                ProcessSelectionProfile::dispatch($selectionProfileId, $extractedPath, $supplierIdGln);
                $this->info("Processing job dispatched to queue: 2ba-selection-profiles");
            } else {
                $this->line("Use --process flag to queue file processing or --process-now to process immediately");
            }

            // Step 6: Delete selection profile from 2BA API
            //$this->deleteSelectionProfile($selectionProfileId);

            // Step 7: Show extracted files
            $this->processExtractedFiles($extractedPath);

            $this->info("SelectionProfile download and extraction completed successfully!");
            $this->line("Extracted files are available at: storage/app/{$extractedPath}");

            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error('Exception occurred: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    private function getDownloadInfo(string $id): ?array
    {
        try {
            $this->line("Getting download information from 2BA API...");

            // Use the connector to get download information
            $response = $this->connector->selectionProfile()->downloadFiles($id);

            if ($response->failed()) {
                $this->error("API request failed with status: {$response->status()}");
                return null;
            }

            $downloadData = $response->dtoOrFail();

            if (empty($downloadData) || !is_array($downloadData)) {
                $this->error('No download information received from API');
                return null;
            }

            // Take the first item from the array
            $downloadInfo = $downloadData[0];

            $this->line("Download information received:");
            $this->line("- ID: " . ($downloadInfo['Id'] ?? 'N/A'));
            $this->line("- FileName: " . ($downloadInfo['FileName'] ?? 'N/A'));
            $this->line("- FileSize: " . $this->formatBytes($downloadInfo['FileSize'] ?? 0));
            $this->line("- IsCompleteSet: " . ($downloadInfo['IsCompleteSet'] ? 'Yes' : 'No'));
            $this->line("- DownloadCount: " . ($downloadInfo['DownloadCount'] ?? 0));

            if (!isset($downloadInfo['Link'])) {
                $this->error('No download link found in API response');
                return null;
            }

            return $downloadInfo;

        } catch (\Exception $e) {
            $this->error("Failed to get download information: " . $e->getMessage());
            return null;
        }
    }

    private function downloadFileFromCdn(array $downloadInfo): ?string
    {
        try {
            $downloadUrl = $downloadInfo['Link'];
            $fileName = $downloadInfo['FileName'] ?? 'selection_profile.zip';

            $this->line("Downloading file from CDN: {$downloadUrl}");

            // Download directly from the CDN URL
            $response = \Illuminate\Support\Facades\Http::timeout(600) // 10 minutes timeout
            ->get($downloadUrl);

            if ($response->failed()) {
                $this->error("CDN download failed with status: {$response->status()}");
                $this->error("Response: {$response->body()}");
                return null;
            }

            // Check content type to ensure it's a ZIP file
            $contentType = $response->header('Content-Type');
            if ($contentType && !str_contains($contentType, 'zip') && !str_contains($contentType, 'octet-stream')) {
                $this->warn("Unexpected content type: {$contentType}");
            }

            // Generate unique filename for temporary storage
            $timestamp = now()->format('Y-m-d_H-i-s');
            $fileId = $downloadInfo['Id'] ?? 'unknown';
            $tempFileName = "2ba_selection_profile_{$fileId}_{$timestamp}.zip";
            $tempPath = "temp/{$tempFileName}";

            // Ensure temp directory exists
            if (!Storage::exists('temp')) {
                Storage::makeDirectory('temp');
            }

            // Store the downloaded file temporarily
            $stored = Storage::put($tempPath, $response->body());

            if (!$stored) {
                $this->error('Failed to store downloaded file to storage');
                return null;
            }

            $fileSize = Storage::size($tempPath);
            $this->line("File stored temporarily at: {$tempPath} (" . $this->formatBytes($fileSize) . ")");
            return $tempPath;

        } catch (\Exception $e) {
            $this->error("Download failed: " . $e->getMessage());
            return null;
        }
    }

    private function extractZipFile(string $tempFilePath): ?string
    {
        try {
            $this->line("Extracting ZIP file...");

            // Get the absolute path to the temporary file
            $absoluteTempPath = Storage::path($tempFilePath);

            if (!file_exists($absoluteTempPath)) {
                $this->error("Temporary file not found: {$absoluteTempPath}");
                return null;
            }

            // Create extraction directory
            $extractPath = "2ba_selection_profiles/" . pathinfo($tempFilePath, PATHINFO_FILENAME);
            $absoluteExtractPath = Storage::path($extractPath);

            // Ensure extraction directory exists
            if (!Storage::exists($extractPath)) {
                Storage::makeDirectory($extractPath);
            }

            // Extract ZIP file
            $zip = new ZipArchive();
            $result = $zip->open($absoluteTempPath);

            if ($result !== TRUE) {
                $this->error("Failed to open ZIP file. Error code: {$result}");
                return null;
            }

            $extractResult = $zip->extractTo($absoluteExtractPath);
            $fileCount = $zip->numFiles;
            $zip->close();

            if (!$extractResult) {
                $this->error('Failed to extract ZIP contents');
                return null;
            }

            $this->line("Successfully extracted {$fileCount} files");
            return $extractPath;

        } catch (\Exception $e) {
            $this->error("Extraction failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Get list of extracted files for further processing
     */
    public function getExtractedFiles(string $extractPath): array
    {
        try {
            return Storage::allFiles($extractPath);
        } catch (\Exception $e) {
            $this->error("Failed to list extracted files: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Helper method to process extracted files (can be extended)
     */
    public function processExtractedFiles(string $extractPath): void
    {
        $files = $this->getExtractedFiles($extractPath);

        $this->line("Found " . count($files) . " extracted files:");

        foreach ($files as $file) {
            $size = Storage::size($file);
            $this->line("  - {$file} (" . $this->formatBytes($size) . ")");
        }
    }

    /**
     * Process directory immediately without using queue (for debugging)
     */
    private function processDirectoryNow(string $selectionProfileId, string $directoryPath, ?string $supplierIdGln): void
    {
        try {
            // Create and execute the job directly
            $job = new ProcessSelectionProfile($selectionProfileId, $directoryPath, $supplierIdGln);
            $job->handle();

        } catch (\Exception $e) {
            $this->error("Error during immediate processing: " . $e->getMessage());
            $this->line("Stack trace:");
            $this->line($e->getTraceAsString());
            throw $e;
        }
    }

    /**
     * Process an existing directory instead of downloading a new profile
     */
    private function processExistingDirectory(string $directoryPath, ?string $supplierIdGln): int
    {
        $this->line("Processing existing directory: {$directoryPath}");

        // Validate that the directory exists
        if (!Storage::exists($directoryPath)) {
            $this->error("Directory not found: {$directoryPath}");
            $this->line("Available directories in storage:");

            // Show available directories for reference
            $directories = Storage::directories('2ba_selection_profiles');
            foreach ($directories as $dir) {
                $this->line("  - {$dir}");
            }

            return Command::FAILURE;
        }

        // Get selection profile ID from config (still needed for processing job)
        $selectionProfileId = config('services.connector.2ba.selection_profile_id', 'existing-directory');

        try {
            // Show extracted files
            $this->processExtractedFiles($directoryPath);

            // Process files based on options
            if ($this->option('process-now')) {
                $this->line("Processing directory immediately (no queue)...");
                $this->processDirectoryNow($selectionProfileId, $directoryPath, $supplierIdGln);
                $this->info("Directory processing completed!");
            } elseif ($this->option('process')) {
                $this->line("Dispatching processing job for existing directory...");
                ProcessSelectionProfile::dispatch($selectionProfileId, $directoryPath, $supplierIdGln);
                $this->info("Processing job dispatched to queue: 2ba-selection-profiles");
            } else {
                $this->line("Use --process flag to queue file processing or --process-now to process immediately");
            }

            $this->info("Existing directory processing completed successfully!");
            $this->line("Directory processed: storage/app/{$directoryPath}");

            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error('Exception occurred while processing existing directory: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    private function deleteSelectionProfile(string $id)
    {
        try {
            $this->line("Delete selectionprofile from 2BA API...");

            // Use the connector to get download information
            $response = $this->connector->selectionProfile()->deleteFile($id);

            if ($response->failed()) {
                $this->error("API request failed with status: {$response->status()}");
                return null;
            }

            return Command::SUCCESS;
        } catch (\Exception $e) {
            $this->error('Exception occurred: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    private function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision) . ' ' . $units[$i];
    }
}
