<?php

use Daylight\Connector2BA\Models\DataItem;
use Daylight\Connector2BA\Models\EtimClass;
use Daylight\Connector2BA\Models\EtimFeature;
use Daylight\Connector2BA\Models\EtimValue;
use Daylight\Connector2BA\Models\EtimUnit;

it('processes unsynced etim data successfully', function () {
    // Create test ETIM class
    $etimClass = EtimClass::create([
        'code' => 'EC000713',
        'name_en' => 'Test Class',
        'name_nl' => 'Test Klasse',
        'description_en' => 'Test description',
        'version' => '10.0',
    ]);

    // Create test ETIM features
    $feature1 = EtimFeature::create([
        'code' => 'EF000064',
        'name_en' => 'Test Feature 1',
        'name_nl' => 'Test Eigenschap 1',
        'datatype' => 'code',
    ]);

    $feature2 = EtimFeature::create([
        'code' => 'EF000227',
        'name_en' => 'Test Feature 2',
        'name_nl' => 'Test Eigenschap 2',
        'datatype' => 'numeric',
        'unit_code' => 'UN001',
    ]);

    $feature3 = EtimFeature::create([
        'code' => 'EF001391',
        'name_en' => 'Test Feature 3',
        'name_nl' => 'Test Eigenschap 3',
        'datatype' => 'numeric',
    ]);

    // Create test ETIM unit
    $unit = EtimUnit::create([
        'code' => 'UN001',
        'symbol' => 'mm',
        'name_en' => 'millimeter',
        'name_nl' => 'millimeter',
    ]);

    // Create test ETIM value
    $value = EtimValue::create([
        'code' => 'EV006013',
        'feature_code' => 'EF000064',
        'label_en' => 'Test Value',
        'label_nl' => 'Test Waarde',
    ]);

    // Create test DataItem with unsynced ETIM data
    $dataItem = DataItem::create([
        'gtin' => '1234567890123',
        'productCode' => 'TEST001',
        'internalCode' => 'INT001',
        'etimClass' => 'EC000713',
        'etim' => [
            'EF000064' => 'EV006013',
            'EF000227' => '800.000',
            'EF001391' => '3.000',
        ],
        'is_synced' => 0,
    ]);

    // Run the command
    $this->artisan('connector:process-etim-data-2ba')
        ->expectsOutput('Starting ETIM data processing...')
        ->expectsOutput('Found 1 items to process.')
        ->expectsOutput("Processing item: {$dataItem->gtin} (Class: {$dataItem->etimClass})")
        ->expectsOutput("✓ Successfully processed: {$dataItem->gtin}")
        ->assertExitCode(0);
});

it('filters by gtin when provided', function () {
    // Create test data with different GTINs
    $dataItem1 = DataItem::create([
        'gtin' => '1111111111111',
        'etimClass' => 'EC000713',
        'etim' => ['EF000064' => 'EV006013'],
        'is_synced' => 0,
    ]);

    $dataItem2 = DataItem::create([
        'gtin' => '2222222222222',
        'etimClass' => 'EC000713',
        'etim' => ['EF000064' => 'EV006013'],
        'is_synced' => 0,
    ]);

    // Run command with GTIN filter
    $this->artisan('connector:process-etim-data-2ba', ['--gtin' => '1111111111111'])
        ->expectsOutput('Filtering by GTIN: 1111111111111')
        ->expectsOutput('Found 1 items to process.')
        ->assertExitCode(0);
});

it('handles no unsynced data gracefully', function () {
    // Create synced data item
    DataItem::create([
        'gtin' => '1234567890123',
        'etimClass' => 'EC000713',
        'etim' => ['EF000064' => 'EV006013'],
        'is_synced' => 1, // Already synced
    ]);

    $this->artisan('connector:process-etim-data-2ba')
        ->expectsOutput('No unsynced ETIM data found.')
        ->assertExitCode(0);
});

it('handles missing etim class gracefully', function () {
    $dataItem = DataItem::create([
        'gtin' => '1234567890123',
        'etimClass' => 'EC999999', // Non-existent class
        'etim' => ['EF000064' => 'EV006013'],
        'is_synced' => 0,
    ]);

    $this->artisan('connector:process-etim-data-2ba')
        ->expectsOutput("✗ Failed to process {$dataItem->gtin}: ETIM Class EC999999 not found")
        ->assertExitCode(0);
});
