<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector2BA\Http\TwoBA;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;

class TestAuthentication extends Command
{
    protected $signature = 'connector:test-auth-2ba {--clear-cache : Clear cached tokens before testing}';

    protected $description = 'Test 2BA OAuth2 authentication and token management';

    public function handle(): int
    {
        if ($this->option('clear-cache')) {
            $this->clearCachedTokens();
            $this->info('Cached tokens cleared');
        }

        $this->line('Testing 2BA OAuth2 authentication...');

        try {
            $connector = new TwoBA();

            // Test 1: Check configuration
            $this->testConfiguration();

            // Test 2: Get access token
            $this->line("\n=== Testing Token Acquisition ===");
            $authenticator = $connector->getAuthenticator();
            $accessToken = $authenticator->refreshToken();
            
            if ($accessToken) {
                $this->info("✓ Successfully obtained access token");
                $this->line("Token (first 20 chars): " . substr($accessToken, 0, 20) . "...");
            } else {
                $this->error("✗ Failed to obtain access token");
                return Command::FAILURE;
            }

            // Test 3: Check cached tokens
            $this->testCachedTokens();

            // Test 4: Test API call
            $this->testApiCall($connector);

            $this->info("\n🎉 All authentication tests passed!");
            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error("Authentication test failed: " . $e->getMessage());
            return Command::FAILURE;
        }
    }

    private function testConfiguration(): void
    {
        $this->line("\n=== Testing Configuration ===");
        
        $config = [
            'base_url' => config('services.connector.2ba.base_url'),
            'client_id' => config('services.connector.2ba.client_id'),
            'client_secret' => config('services.connector.2ba.client_secret'),
            'username' => config('services.connector.2ba.username'),
            'password' => config('services.connector.2ba.password'),
        ];

        foreach ($config as $key => $value) {
            if (empty($value)) {
                $this->error("✗ Missing configuration: CONNECTOR_2BA_" . strtoupper($key));
            } else {
                if ($key === 'password' || $key === 'client_secret') {
                    $displayValue = str_repeat('*', min(strlen($value), 8));
                } else {
                    $displayValue = $value;
                }
                $this->info("✓ {$key}: {$displayValue}");
            }
        }
    }

    private function testCachedTokens(): void
    {
        $this->line("\n=== Testing Cached Tokens ===");
        
        $accessToken = Cache::get('connector_2ba_access_token');
        $refreshToken = Cache::get('connector_2ba_refresh_token');

        if ($accessToken) {
            $expiresAt = $accessToken['expires_at'] ?? 'unknown';
            $remaining = is_numeric($expiresAt) ? max(0, $expiresAt - time()) : 'unknown';
            $this->info("✓ Access token cached (expires in {$remaining} seconds)");
        } else {
            $this->warn("! No access token in cache");
        }

        if ($refreshToken) {
            $this->info("✓ Refresh token cached");
        } else {
            $this->warn("! No refresh token in cache");
        }
    }

    private function testApiCall(TwoBA $connector): void
    {
        $this->line("\n=== Testing API Call ===");
        
        try {
            // Test a simple API call - you might need to adjust the endpoint
            $response = $connector->selectionProfile()->downloadFiles('test');
            
            // We expect this to fail with a 404 or similar, but it should be authenticated
            if ($response->status() === 404) {
                $this->info("✓ API call authenticated (404 - endpoint not found is expected)");
            } elseif ($response->status() === 401) {
                $this->error("✗ API call not authenticated (401)");
            } elseif ($response->successful()) {
                $this->info("✓ API call successful");
            } else {
                $this->warn("? API call returned status: " . $response->status());
            }

        } catch (\Exception $e) {
            $this->error("✗ API call failed: " . $e->getMessage());
        }
    }

    private function clearCachedTokens(): void
    {
        Cache::forget('connector_2ba_access_token');
        Cache::forget('connector_2ba_refresh_token');
    }
}
