<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\Connector2BA\Http\TwoBA;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\Connector2BA\Jobs\UpdateProduct;

class GetDataByGTIN extends Command
{
    protected $signature = 'connector:get-data-by-gtin {--force} {--gtin=}';

    protected $description = 'Get product data from 2BA by GTIN';

    protected BaseConnector|TwoBA $connector;

    public function handle(): int
    {
        $this->connector = new TwoBA();

        if (!method_exists($this->connector, 'getProductMapper')) {
            $this->error('No products mapper found in ' . $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start product enrichtment for ' . $this->connector->getName());

        $page = 1;
        $totalProcessed = 0;

        do {
            try {
                $results = $this->getItems($page);

                if ($results->isEmpty()) {
                    break;
                }

                dd($results);

                $jobs = $results->map(fn($product) => new UpdateProduct($product))->toArray();

                Bus::batch($jobs)
                    ->onQueue('2ba-products')
                    ->dispatch();

                $totalProcessed += $results->count();
                $this->line("-> Dispatched {$results->count()} productdata jobs for page {$page}");

                $page++;

                // Prevent memory leaks
                unset($jobs, $results);
                gc_collect_cycles();

            } catch (\Exception $e) {
                $this->error('Exception: ' . $e->getMessage());
                break;
            }
        } while (true);

        $this->info("Total products processed: {$totalProcessed}");

        return Command::SUCCESS;
    }

    protected function getItems(): Collection
    {
        try {
            return $this->connector
                ->products()
                ->get($this->connector->getProductMapper(), $this->option('gtin'))
                ->dtoOrFail();
        } catch (\Exception $e) {
            $this->error("Failed to fetch products: " . $e->getMessage());
            return collect();
        }
    }
}

