<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector2BA\Models\DataItem;
use Daylight\Connector2BA\Service\AttachmentCSVReader;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\LazyCollection;

/**
 * Import attachements from 2BA Selectrion Profile to DataItem attachments, split up to pdf and image
 *
 * @examples
 * - php artisan connector:import-attachments-2ba
 * - php artisan connector:import-attachments-2ba --directory="2ba_selection_profiles/2ba_selection_profile_dc750459-bf8d-4f8d-a8fc-a7a27f08e0d4_2025-11-02_20-28-17"
 * - php artisan connector:import-attachments-2ba --batch-size=500
 */

class ImportAttachments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'connector:import-attachments-2ba
                            {--directory= : Specify a specific directory instead of using the newest one}
                            {--batch-size=1000 : Number of records to process per batch}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import attachments from AttachmentConversion.csv to DataItem records';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Starting attachment import...');

        try {
            // Find the CSV file
            $csvPath = $this->findCsvFile();

            if (!$csvPath) {
                $this->error('AttachmentConversion.csv not found in 2ba_selection_profiles directory');
                return Command::FAILURE;
            }

            $this->info("Found CSV file: {$csvPath}");

            // Initialize reader
            $reader = new AttachmentCSVReader();
            $batchSize = (int) $this->option('batch-size');

            // Group attachments by productCode
            $attachmentsByProduct = $this->groupAttachmentsByProduct($reader->readFile($csvPath));

            $this->info("Processing " . count($attachmentsByProduct) . " products with attachments...");

            // Process in batches
            $this->processBatches($attachmentsByProduct, $batchSize);

            $this->info('Attachment import completed successfully!');

            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error('Exception occurred: ' . $e->getMessage());
            $this->line($e->getTraceAsString());
            return Command::FAILURE;
        }
    }

    /**
     * Find the AttachmentConversion.csv file.
     *
     * @return string|null Full path to the CSV file or null if not found
     */
    private function findCsvFile(): ?string
    {
        $basePath = '2ba_selection_profiles';

        // If directory option is provided, use that
        if ($this->option('directory')) {
            $directory = $this->option('directory');
            $csvPath = $directory . '/AttachmentConversion.csv';

            if (Storage::exists($csvPath)) {
                return Storage::path($csvPath);
            }

            $this->error("CSV file not found in specified directory: {$directory}");
            return null;
        }

        // Find newest directory
        if (!Storage::exists($basePath)) {
            $this->error("Directory not found: {$basePath}");
            return null;
        }

        $directories = Storage::directories($basePath);

        if (empty($directories)) {
            $this->error("No directories found in {$basePath}");
            return null;
        }

        // Sort directories by name (assuming they contain timestamps)
        // Newest should be last alphabetically if they follow the pattern
        usort($directories, function ($a, $b) {
            return strcmp($b, $a); // Reverse sort for newest first
        });

        $newestDirectory = $directories[0];
        $this->line("Using directory: {$newestDirectory}");

        $csvPath = $newestDirectory . '/AttachmentConversion.csv';

        if (!Storage::exists($csvPath)) {
            $this->error("AttachmentConversion.csv not found in {$newestDirectory}");
            return null;
        }

        return Storage::path($csvPath);
    }

    /**
     * Group attachments by productCode and type.
     *
     * @param LazyCollection $attachments
     * @return array Array keyed by productCode with grouped attachments
     */
    private function groupAttachmentsByProduct(LazyCollection $attachments): array
    {
        $grouped = [];

        foreach ($attachments as $attachment) {
            $productCode = $attachment['productCode'];
            $type = $attachment['type'];
            $location = $attachment['location'];
            $mutationDate = $attachment['mutationDate'];

            if (!isset($grouped[$productCode])) {
                $grouped[$productCode] = [
                    'images' => [],
                    'pdfs' => [],
                ];
            }

            $attachmentData = [
                'location' => $location,
                'mutation_date' => $mutationDate,
            ];

            if ($type === 'image') {
                $grouped[$productCode]['images'][] = $attachmentData;
            } elseif ($type === 'pdf') {
                $grouped[$productCode]['pdfs'][] = $attachmentData;
            }
        }

        return $grouped;
    }

    /**
     * Process attachments in batches and update DataItem records.
     *
     * @param array $attachmentsByProduct
     * @param int $batchSize
     */
    private function processBatches(array $attachmentsByProduct, int $batchSize): void
    {
        $total = count($attachmentsByProduct);
        $processed = 0;
        $updated = 0;
        $skipped = 0;

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $chunks = array_chunk($attachmentsByProduct, $batchSize, true);

        foreach ($chunks as $chunk) {
            foreach ($chunk as $productCode => $attachments) {
                try {
                    // Find existing DataItem
                    $dataItem = DataItem::where('productCode', $productCode)->first();

                    if (!$dataItem) {
                        $skipped++;
                        $bar->advance();
                        continue;
                    }

                    // Replace attachments (not merge)
                    $dataItem->attachments = $attachments;
                    $dataItem->sync_media_synced = 0;
                    $dataItem->save();

                    $updated++;
                } catch (\Exception $e) {
                    $this->newLine();
                    $this->error("Failed to update productCode {$productCode}: " . $e->getMessage());
                    $skipped++;
                }

                $processed++;
                $bar->advance();
            }

            // Memory management
            gc_collect_cycles();
        }

        $bar->finish();
        $this->newLine(2);

        $this->info("Processed: {$processed} products");
        $this->info("Updated: {$updated} DataItems");
        $this->info("Skipped: {$skipped} products (not found in database)");
    }
}


