<?php

namespace Daylight\Connector2BA\Http;

use Daylight\Connector\Contracts\MapperInterface;
use Daylight\Connector\Services\BaseConnector;
use Daylight\Connector\Support\ConnectorLogger;
use Daylight\Connector2BA\Http\Auth\OAuth2Authenticator;
use Daylight\Connector2BA\Mappers\ProductMapper;
use Daylight\Connector2BA\Resources\Products;
use Daylight\Connector2BA\Resources\SelectionProfile;
use Illuminate\Support\Str;

/**
 * Connector for 2BA system.
 *
 * @source: https://api.2ba.nl/1/docs/index.html?url=/1/docs/swagger.json#/
 */

class TwoBA extends BaseConnector
{

    public function __construct()
    {
        parent::__construct(
            new ConnectorLogger()
        );
    }

    /**
     * Get the name of the connector.
     */
    public function getName(): string
    {
        return '2BA';
    }

    public function getKey(): string
    {
        return Str::slug($this->getName());
    }

    public function resolveBaseUrl(): string
    {
        return config('services.connector.2ba.base_url');
    }

    public function defaultHeaders(): array
    {
        return [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'Cache-Control' => 'no-cache',
        ];
    }

    protected function defaultAuth(): OAuth2Authenticator
    {
        return new OAuth2Authenticator(
            config('services.connector.2ba.client_id'),
            config('services.connector.2ba.client_secret'),
            config('services.connector.2ba.username'),
            config('services.connector.2ba.password')
        );
    }

    public function products(): \Daylight\Connector2BA\Resources\Products
    {
        return new Products($this);
    }

    public function selectionProfile(): SelectionProfile
    {
        return new SelectionProfile($this);
    }

    public function getProductMapper(): MapperInterface
    {
        return new ProductMapper();
    }

    /**
     * Get the authenticator instance (for testing purposes)
     */
    public function getAuthenticator(): OAuth2Authenticator
    {
        return $this->defaultAuth();
    }
}
