<?php

namespace Daylight\Connector2BA\Jobs;

use Daylight\Connector2BA\Entities\Product as ProductEntity;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class UpdateProduct implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        private ProductEntity $product
    ) {
        $this->onQueue('2ba-products');
    }

    public function handle(): void
    {
        try {
            // Here you would typically save the product data to your local database
            // Focus on marketing content, attributes, and media from 2BA
            
            Log::info('Processing 2BA product enrichment data', [
                'id' => $this->product->id,
                'sku' => $this->product->sku,
                'name' => $this->product->name,
                'has_marketing_text' => !empty($this->product->marketingText),
                'has_images' => $this->product->hasImages(),
                'has_documents' => $this->product->hasDocuments(),
                'has_videos' => $this->product->hasVideos(),
                'specifications_count' => $this->product->specifications ? count($this->product->specifications) : 0,
            ]);

            // Example implementation for enriching existing products:
            // $localProduct = Product::where('sku', $this->product->sku)->first();
            // 
            // if ($localProduct) {
            //     $localProduct->update([
            //         'short_description' => $this->product->shortDescription,
            //         'long_description' => $this->product->longDescription,
            //         'marketing_text' => $this->product->marketingText,
            //         'technical_description' => $this->product->technicalDescription,
            //         'model' => $this->product->model,
            //         'color' => $this->product->color,
            //         'size' => $this->product->size,
            //         'material' => $this->product->material,
            //         'weight' => $this->product->weight,
            //         'dimensions' => $this->product->dimensions,
            //         'specifications' => $this->product->specifications,
            //         'features' => $this->product->features,
            //         'attributes' => $this->product->attributes,
            //     ]);
            //     
            //     // Process media separately
            //     $this->processProductMedia($localProduct);
            // }

        } catch (\Exception $e) {
            Log::error('Failed to process 2BA product enrichment', [
                'product_id' => $this->product->id,
                'sku' => $this->product->sku,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    private function processProductMedia($localProduct): void
    {
        // Process images
        if ($this->product->hasImages()) {
            foreach ($this->product->images as $imageData) {
                // Save image to media library
                // $localProduct->addMediaFromUrl($imageData['url'])
                //     ->withCustomProperties(['alt' => $imageData['alt']])
                //     ->toMediaCollection('images');
            }
        }

        // Process documents
        if ($this->product->hasDocuments()) {
            foreach ($this->product->documents as $documentData) {
                // Save document to media library
                // $localProduct->addMediaFromUrl($documentData['url'])
                //     ->withCustomProperties(['title' => $documentData['title']])
                //     ->toMediaCollection('documents');
            }
        }

        // Process videos
        if ($this->product->hasVideos()) {
            foreach ($this->product->videos as $videoData) {
                // Save video data
                // $localProduct->videos()->create([
                //     'url' => $videoData['url'],
                //     'title' => $videoData['title'],
                //     'thumbnail' => $videoData['thumbnail'],
                // ]);
            }
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('2BA Product job failed permanently', [
            'product_id' => $this->product->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
