<?php

require_once __DIR__ . '/vendor/autoload.php';

use Daylight\Connector2BA\Service\ETIMReader;

// Function to get current memory usage in MB
function getMemoryUsage() {
    return round(memory_get_usage(true) / 1024 / 1024, 2);
}

function getPeakMemoryUsage() {
    return round(memory_get_peak_usage(true) / 1024 / 1024, 2);
}

// Create a larger test JSON file to simulate memory pressure
function createLargeTestFile($filePath, $itemCount = 1000) {
    $handle = fopen($filePath, 'w');
    if (!$handle) {
        throw new RuntimeException("Cannot create test file: $filePath");
    }
    
    fwrite($handle, "[\n");
    
    for ($i = 0; $i < $itemCount; $i++) {
        $item = [
            'gtin' => '123456789012' . str_pad($i, 3, '0', STR_PAD_LEFT),
            'tradeItemId' => 'SUPPLIER' . str_pad($i, 6, '0', STR_PAD_LEFT),
            'supplierGLN' => '1234567890123',
            'description' => 'Test Product ' . $i . ' - ' . str_repeat('Lorem ipsum dolor sit amet consectetur adipiscing elit ', 10),
            'manufacturerGLN' => '9876543210987',
            'manufacturerArticleCode' => 'MFG' . str_pad($i, 6, '0', STR_PAD_LEFT),
            'orderable' => ($i % 2 === 0),
            'processable' => true,
            'status' => ($i % 3 === 0) ? 'active' : 'discontinued',
            'weight' => round(1.5 + ($i * 0.1), 2),
            'height' => round(10.0 + ($i * 0.05), 2),
            'width' => round(5.0 + ($i * 0.03), 2),
            'depth' => round(8.0 + ($i * 0.04), 2),
            'minimumOrderQuantity' => max(1, $i % 10),
            'orderMultiple' => max(1, $i % 5),
            'deliveryTime' => 7 + ($i % 14),
            'grossPrice' => round(99.99 + ($i * 1.50), 2),
            'netPrice' => round(85.00 + ($i * 1.25), 2),
            'currency' => 'EUR',
            'vatPercentage' => 21.0,
            'additionalData' => [
                'category' => 'Category ' . ($i % 20),
                'subcategory' => 'Subcategory ' . ($i % 50),
                'features' => array_fill(0, 10, 'Feature ' . $i),
                'specifications' => [
                    'color' => 'Color ' . ($i % 10),
                    'material' => 'Material ' . ($i % 15),
                    'size' => 'Size ' . ($i % 8),
                ]
            ]
        ];
        
        fwrite($handle, json_encode($item, JSON_PRETTY_PRINT));
        if ($i < $itemCount - 1) {
            fwrite($handle, ",\n");
        }
    }
    
    fwrite($handle, "\n]");
    fclose($handle);
}

// Create test directory and file
$testDir = __DIR__ . '/test_data';
if (!is_dir($testDir)) {
    mkdir($testDir, 0755, true);
}

$testFile = $testDir . '/LargeETIMxChangeExport.json';
$itemCount = 5000; // Create 5000 items for testing

echo "Creating large test JSON file with $itemCount items...\n";
echo "Memory before file creation: " . getMemoryUsage() . " MB\n";

createLargeTestFile($testFile, $itemCount);

$fileSize = round(filesize($testFile) / 1024 / 1024, 2);
echo "Test file created: $testFile ($fileSize MB)\n";
echo "Memory after file creation: " . getMemoryUsage() . " MB\n\n";

// Test the memory-optimized ETIMReader
try {
    echo "Testing memory-optimized ETIMReader:\n";
    echo "====================================\n";
    
    $memoryBefore = getMemoryUsage();
    echo "Memory before processing: $memoryBefore MB\n";
    
    $reader = new ETIMReader($testDir, 'LargeETIMxChangeExport.json');
    
    $batchNumber = 1;
    $totalProcessed = 0;
    $maxMemoryDuringProcessing = $memoryBefore;
    
    foreach ($reader->readFile(100) as $batch) {
        $currentMemory = getMemoryUsage();
        $maxMemoryDuringProcessing = max($maxMemoryDuringProcessing, $currentMemory);
        
        echo "Batch $batchNumber: " . count($batch) . " items, Memory: {$currentMemory} MB\n";
        
        // Verify first item in first batch
        if ($batchNumber === 1 && !empty($batch)) {
            $firstItem = $batch[0];
            echo "  First item GTIN: " . ($firstItem['GTIN'] ?? 'N/A') . "\n";
            echo "  First item Description preview: " . substr($firstItem['Description'] ?? 'N/A', 0, 50) . "...\n";
        }
        
        $totalProcessed += count($batch);
        $batchNumber++;
        
        // Show progress every 10 batches
        if ($batchNumber % 10 === 0) {
            echo "  Progress: $totalProcessed items processed\n";
        }
    }
    
    $memoryAfter = getMemoryUsage();
    $peakMemory = getPeakMemoryUsage();
    
    echo "\nProcessing completed!\n";
    echo "Total items processed: $totalProcessed\n";
    echo "Memory before: $memoryBefore MB\n";
    echo "Memory after: $memoryAfter MB\n";
    echo "Peak memory usage: $peakMemory MB\n";
    echo "Max memory during processing: $maxMemoryDuringProcessing MB\n";
    echo "Memory increase: " . round($maxMemoryDuringProcessing - $memoryBefore, 2) . " MB\n";
    
    // Calculate efficiency
    $memoryPerItem = round(($maxMemoryDuringProcessing - $memoryBefore) / $totalProcessed * 1024, 2);
    echo "Memory per item: {$memoryPerItem} KB\n";
    
    if ($maxMemoryDuringProcessing - $memoryBefore < 50) {
        echo "✅ Memory optimization SUCCESS: Low memory usage increase\n";
    } else {
        echo "⚠️  Memory usage higher than expected\n";
    }
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

// Clean up test file
unlink($testFile);
rmdir($testDir);
echo "\nTest completed and cleaned up.\n";

