<?php

require_once __DIR__ . '/vendor/autoload.php';

use Daylight\Connector2BA\Service\ETIMReader;

// Create test data with the actual ETIM structure
$etimData = [
    "CatalogueId" => "2BA-EXP-1299151160",
    "CatalogueName" => [
        [
            "Language" => "nl-NL",
            "CatalogueName" => "2BA Export: HEG klantportaal 2025-09-10"
        ]
    ],
    "CatalogueVersion" => "1",
    "CatalogueType" => "FULL",
    "GenerationDate" => "2025-09-10",
    "CatalogueValidityStart" => "2025-09-10",
    "Language" => ["nl-NL"],
    "Supplier" => [
        [
            "SupplierName" => "Horst Electro Grossier (HEG)",
            "SupplierIdGln" => "8714253038889",
            "Product" => [
                [
                    "ProductIdentification" => [
                        "ManufacturerIdGln" => "0042186000016",
                        "ManufacturerName" => "Zircon",
                        "ManufacturerProductNumber" => "68720",
                        "ProductGtin" => ["04218668721007"]
                    ],
                    "TradeItem" => [
                        [
                            "ItemIdentification" => [
                                "SupplierItemNumber" => "6028665004",
                                "ItemGtin" => ["04218668721007"],
                                "DiscountGroupId" => "60171"
                            ],
                            "ItemDetails" => [
                                "ItemStatus" => "ACTIVE",
                                "StockItem" => true,
                                "ItemDescriptions" => [
                                    [
                                        "DescriptionLanguage" => "nl-NL",
                                        "MinimalItemDescription" => "ZIRCON L550C MULTIFUNCT. MUURSCANNER"
                                    ]
                                ]
                            ],
                            "ItemLogisticDetails" => [[]],
                            "Ordering" => [
                                "OrderUnit" => "C62",
                                "MinimumOrderQuantity" => "1.000",
                                "OrderStepSize" => "1.000",
                                "UseUnit" => "C62",
                                "UseUnitConversionFactor" => "1.000",
                                "SingleUseUnitQuantity" => "1.000"
                            ],
                            "Pricing" => [
                                [
                                    "PriceUnit" => "C62",
                                    "PriceUnitFactor" => "1.000",
                                    "PriceQuantity" => "1.000",
                                    "PriceOnRequest" => false,
                                    "GrossListPrice" => "92.000",
                                    "Vat" => "21.00",
                                    "PriceValidityDate" => "2023-08-02"
                                ]
                            ],
                            "ItemCountrySpecificFields" => [
                                [
                                    "CSItemCharacteristicCode" => "NL_012",
                                    "CSItemCharacteristicName" => [
                                        [
                                            "CSItemCharacteristicName" => "Orderable indicator"
                                        ]
                                    ],
                                    "CSItemCharacteristicValueBoolean" => true
                                ],
                                [
                                    "CSItemCharacteristicCode" => "NL_014",
                                    "CSItemCharacteristicName" => [
                                        [
                                            "CSItemCharacteristicName" => "Fragile on package"
                                        ]
                                    ],
                                    "CSItemCharacteristicValueBoolean" => false
                                ],
                                [
                                    "CSItemCharacteristicCode" => "NL_015",
                                    "CSItemCharacteristicName" => [
                                        [
                                            "CSItemCharacteristicName" => "Best before date on package"
                                        ]
                                    ],
                                    "CSItemCharacteristicValueBoolean" => false
                                ],
                                [
                                    "CSItemCharacteristicCode" => "NL_016-1",
                                    "CSItemCharacteristicName" => [
                                        [
                                            "CSItemCharacteristicName" => "Cuttable indicator"
                                        ]
                                    ],
                                    "CSItemCharacteristicValueBoolean" => false
                                ]
                            ]
                        ]
                    ]
                ],
                // Add a second product for testing
                [
                    "ProductIdentification" => [
                        "ManufacturerIdGln" => "0042186000016",
                        "ManufacturerName" => "Zircon",
                        "ManufacturerProductNumber" => "68721",
                        "ProductGtin" => ["04218668721014"]
                    ],
                    "TradeItem" => [
                        [
                            "ItemIdentification" => [
                                "SupplierItemNumber" => "6028665005",
                                "ItemGtin" => ["04218668721014"],
                                "DiscountGroupId" => "60171"
                            ],
                            "ItemDetails" => [
                                "ItemStatus" => "DISCONTINUED",
                                "StockItem" => false,
                                "ItemDescriptions" => [
                                    [
                                        "DescriptionLanguage" => "nl-NL",
                                        "MinimalItemDescription" => "ZIRCON L550D ADVANCED MUURSCANNER"
                                    ]
                                ]
                            ],
                            "ItemLogisticDetails" => [[]],
                            "Ordering" => [
                                "OrderUnit" => "C62",
                                "MinimumOrderQuantity" => "5.000",
                                "OrderStepSize" => "5.000",
                                "UseUnit" => "C62",
                                "UseUnitConversionFactor" => "1.000",
                                "SingleUseUnitQuantity" => "1.000"
                            ],
                            "Pricing" => [
                                [
                                    "PriceUnit" => "C62",
                                    "PriceUnitFactor" => "1.000",
                                    "PriceQuantity" => "1.000",
                                    "PriceOnRequest" => false,
                                    "GrossListPrice" => "125.000",
                                    "Vat" => "21.00",
                                    "PriceValidityDate" => "2023-08-02"
                                ]
                            ],
                            "ItemCountrySpecificFields" => [
                                [
                                    "CSItemCharacteristicCode" => "NL_012",
                                    "CSItemCharacteristicName" => [
                                        [
                                            "CSItemCharacteristicName" => "Orderable indicator"
                                        ]
                                    ],
                                    "CSItemCharacteristicValueBoolean" => false
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ]
    ]
];

// Create test directory and file
$testDir = __DIR__ . '/test_data';
if (!is_dir($testDir)) {
    mkdir($testDir, 0755, true);
}

$testFile = $testDir . '/ETIMxChangeExport.json';
file_put_contents($testFile, json_encode($etimData, JSON_PRETTY_PRINT));

echo "Created ETIM test JSON file: $testFile\n";
echo "Expected 2 TradeItems from 2 Products under 1 Supplier\n\n";

// Test the ETIMReader
try {
    $reader = new ETIMReader($testDir, 'ETIMxChangeExport.json');
    
    echo "Testing ETIMReader with ETIM structure:\n";
    echo "======================================\n";
    
    $batchNumber = 1;
    $totalProcessed = 0;
    
    foreach ($reader->readFile(10) as $batch) {
        echo "Batch $batchNumber:\n";
        echo "  Items in batch: " . count($batch) . "\n";
        
        foreach ($batch as $index => $item) {
            echo "  Item " . ($index + 1) . ":\n";
            echo "    TradeItemId: " . ($item['TradeItemId'] ?? 'N/A') . "\n";
            echo "    GTIN: " . ($item['GTIN'] ?? 'N/A') . "\n";
            echo "    ProductGTIN: " . ($item['ProductGTIN'] ?? 'N/A') . "\n";
            echo "    Description: " . ($item['Description'] ?? 'N/A') . "\n";
            echo "    SupplierGLN: " . ($item['SupplierGLN'] ?? 'N/A') . "\n";
            echo "    SupplierName: " . ($item['SupplierName'] ?? 'N/A') . "\n";
            echo "    ManufacturerName: " . ($item['ManufacturerName'] ?? 'N/A') . "\n";
            echo "    StatusCode: " . ($item['StatusCode'] ?? 'N/A') . "\n";
            echo "    OrderableIndicator: " . ($item['OrderableIndicator'] ?? 'N/A') . "\n";
            echo "    ProcessableIndicator: " . ($item['ProcessableIndicator'] ?? 'N/A') . "\n";
            echo "    MinimumOrderQuantity: " . ($item['MinimumOrderQuantity'] ?? 'N/A') . "\n";
            echo "    GrossPriceInOrderUnit: " . ($item['GrossPriceInOrderUnit'] ?? 'N/A') . "\n";
            echo "    VatPercentage: " . ($item['VatPercentage'] ?? 'N/A') . "\n";
            echo "    Source file: " . ($item['_source_file'] ?? 'N/A') . "\n";
            echo "    Item number: " . ($item['_item_number'] ?? 'N/A') . "\n";
            echo "\n";
        }
        
        $totalProcessed += count($batch);
        $batchNumber++;
    }
    
    echo "Total items processed: $totalProcessed\n";
    
    if ($totalProcessed === 2) {
        echo "✅ SUCCESS: Correctly extracted 2 TradeItems from ETIM structure\n";
    } else {
        echo "❌ ERROR: Expected 2 items, got $totalProcessed\n";
    }
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

// Clean up test file
unlink($testFile);
rmdir($testDir);
echo "\nTest completed and cleaned up.\n";

