# Daylight Connector 2BA

Een Laravel package voor integratie met het 2BA productdatabase systeem. 2BA is een dataset met aanvullende productgegevens zoals marketingteksten, productattributen, specificaties en media (afbeeldingen, documenten, video's).

## Installatie

De package wordt automatisch geladen via Composer en de Laravel service provider.

## Configuratie

Voeg de volgende environment variabelen toe aan je `.env` bestand:

```env
CONNECTOR_2BA_BASE_URL=https://api.2ba.nl
CONNECTOR_2BA_CLIENT_ID=your_client_id
CONNECTOR_2BA_CLIENT_SECRET=your_client_secret
CONNECTOR_2BA_USERNAME=your_2ba_username
CONNECTOR_2BA_PASSWORD=your_2ba_password
CONNECTOR_2BA_SELECTION_PROFILE_ID=your_selection_profile_id
```

## Gebruik

### Console Commands

De package registreert automatisch de volgende Artisan commands:

```bash
# Productverrijkingsdata synchroniseren
php artisan connector:update-products-2ba

# SelectionProfile bestand downloaden en uitpakken
php artisan connector:download-selection-profile-2ba

# SelectionProfile downloaden met automatische verwerking
php artisan connector:download-selection-profile-2ba --process

# Test OAuth2 authenticatie
php artisan connector:test-auth-2ba

# Test authenticatie met cache clear
php artisan connector:test-auth-2ba --clear-cache
```

### Programmatisch gebruik

```php
use Daylight\Connector2BA\Http\TwoBA;

$connector = new TwoBA();

// Productverrijkingsdata ophalen
$products = $connector
    ->products()
    ->all($connector->getProductMapper(), 1, 100);

foreach ($products->dtoOrFail() as $product) {
    echo "Product: " . $product->name . "\n";
    echo "Marketing tekst: " . $product->marketingText . "\n";
    echo "Heeft afbeeldingen: " . ($product->hasImages() ? 'Ja' : 'Nee') . "\n";
    echo "Specificaties: " . count($product->specifications ?? []) . "\n";
}

// SelectionProfile bestand downloaden
$response = $connector->selectionProfile()->downloadFiles('12345');
if ($response->successful()) {
    // Verwerk de gedownloade ZIP data
    $zipContent = $response->body();
}
```

## SelectionProfile Downloads

De 2BA connector ondersteunt het downloaden van SelectionProfile bestanden:

### Download Command
```bash
# Download en extraheer SelectionProfile (gebruikt geconfigureerde ID)
php artisan connector:download-selection-profile-2ba

# Download, extraheer en start verwerking
php artisan connector:download-selection-profile-2ba --process
```

### Wat gebeurt er:
1. **API Call**: Vraag downloadinformatie op van `/SelectionProfile/filesB`
2. **CDN Download**: Download ZIP bestand van de geretourneerde CDN link  
3. **Tijdelijke opslag**: Bestand wordt tijdelijk opgeslagen in `storage/app/temp/`
4. **Extractie**: ZIP wordt uitgepakt naar `storage/app/2ba_selection_profiles/`
5. **Cleanup**: Tijdelijk ZIP bestand wordt verwijderd
6. **Verwerking** (optioneel): Queue job voor bestandsverwerking

### Bestandsstructuur
Na extractie vindt je de bestanden in:
```
storage/app/2ba_selection_profiles/
└── 2ba_selection_profile_{id}_{timestamp}/
    ├── products.xml
    ├── media.csv
    ├── specifications.json
    └── ...
```

### Verwerking
Met de `--process` flag wordt automatisch een queue job gestart die:
- Alle geëxtraheerde bestanden analyseert
- Bestanden verwerkt op basis van type (XML, CSV, JSON, TXT)
- Logging bijhoudt van de verwerking
- Fouten afhandelt per bestand

## Authenticatie

De connector gebruikt OAuth2 Resource Owner Password Credentials Grant voor authenticatie volgens de [2BA OAuth2 specificatie](https://www.2ba.nl/documentatie/webservices/oauth2/). 

### OAuth2 Flow:
1. **Password Grant**: Verkrijg initial access token en refresh token met gebruikerscredentials
2. **Token Refresh**: Automatische refresh van access tokens met refresh token  
3. **Caching**: Access tokens (120 sec) en refresh tokens (30 dagen) worden gecached
4. **Bearer Authorization**: Access token wordt gebruikt als Bearer token in API calls

### Token Management:
- Access tokens verlopen na 120 seconden (zoals gespecificeerd door 2BA)
- Refresh tokens worden automatisch gebruikt om nieuwe access tokens te verkrijgen
- Bij falen van refresh token wordt opnieuw een password grant uitgevoerd
- Tokens worden veilig gecached in Laravel's cache systeem

## Data Mapping

De package bevat een mapper voor productverrijkingsdata:

- **Products**: 
  - Basis informatie: ID, SKU, EAN, naam
  - Beschrijvingen: Korte beschrijving, lange beschrijving, marketingtekst, technische beschrijving
  - Attributen: Merk, model, kleur, maat, materiaal, afmetingen, gewicht
  - Specificaties: Technische specificaties en kenmerken
  - Media: Afbeeldingen, documenten, video's met metadata

## Queue Jobs

Data verwerking wordt asynchroon verwerkt via Laravel queues:

- `2ba-products` queue voor product enrichment  
- `2ba-selection-profiles` queue voor SelectionProfile verwerking

Zorg ervoor dat queue workers draaien:

```bash
php artisan queue:work --queue=2ba-products,2ba-selection-profiles
```

## Scheduled Tasks

Het command wordt automatisch gescheduled:

- Productverrijking: Dagelijks om 02:00

## API Endpoints

De connector verwacht de volgende 2BA API endpoints:

- `GET /api/v1/products` - Productverrijkingsdata lijst
- `GET /SelectionProfile/filesB?id={id}` - SelectionProfile download informatie
- `POST /oauth/token` - OAuth2 token endpoint

### SelectionProfile API Response
Het `/SelectionProfile/filesB` endpoint retourneert een JSON array met downloadinformatie:
```json
[
  {
    "Id": "58bd43a4-b137-41e8-89cb-512007317d75",
    "MutationsFromDate": "/Date(1387470343000+0200)/",
    "MutationsToDate": "/Date(1387470343000+0200)/",
    "ExportReadyDate": "/Date(1387470343000+0100)/",
    "Link": "https://cdn.2ba.nl/downloads/get/?fileid=bb08c2b5-0057-4712-b66b-df73282c756a",
    "IsCompleteSet": true,
    "Summary": "null",
    "FileName": "P200_agp-acces_20131219_172100.zip",
    "FileSize": 0,
    "DownloadCount": 5
  }
]
```

De `Link` property bevat de CDN URL waar het daadwerkelijke ZIP bestand gedownload kan worden.

## Foutafhandeling

- API fouten worden gelogd via Laravel's logging systeem
- Failed jobs worden automatisch geretried
- OAuth2 tokens worden automatisch vernieuwd bij expiratie

## Productverrijking

Deze connector is specifiek ontworpen voor het verrijken van bestaande productdata met:

### Marketingcontent
- Korte en lange productbeschrijvingen
- Marketingteksten voor webshops
- Technische beschrijvingen

### Media & Assets
- Productafbeeldingen met alt-tekst en volgorde
- Productdocumenten (handleidingen, datasheets)
- Productvideo's met thumbnails

### Specificaties & Attributen
- Technische specificaties
- Productkenmerken
- Afmetingen en gewicht
- Materialen en kleuren

### Gebruik in E-commerce
```php
// Voorbeeld: verrijk bestaand product
$existingProduct = Product::where('sku', $enrichmentData->sku)->first();

if ($existingProduct) {
    $existingProduct->update([
        'marketing_text' => $enrichmentData->marketingText,
        'technical_description' => $enrichmentData->technicalDescription,
        'specifications' => $enrichmentData->specifications,
    ]);
    
    // Voeg afbeeldingen toe
    foreach ($enrichmentData->images as $image) {
        $existingProduct->addMediaFromUrl($image['url'])
            ->toMediaCollection('gallery');
    }
}
```
php artisan connector:download-selection-profile-2ba --directory="2ba_selection_profiles/2ba_selection_profile_41881231-f77e-43a6-99dd-cb5848693c4a_2025-09-22_10-17-45" --process
php artisan connector:download-selection-profile-2ba --directory="2ba_selection_profiles/2ba_selection_profile_41881231-f77e-43a6-99dd-cb5848693c4a_2025-09-22_10-17-45" --process-now
