<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // 1. Klassen (EC)
        Schema::create('etim_classes', function (Blueprint $table) {
            $table->string('code')->primary(); // EC-code, bv. EC001097
            $table->string('name_en');
            $table->string('name_nl')->nullable();
            $table->text('description_en')->nullable();
            $table->text('description_nl')->nullable();
            $table->string('sector')->nullable();
            $table->string('version')->default('10.0');
            $table->timestamps();
        });

        // 2. Features (EF)
        Schema::create('etim_features', function (Blueprint $table) {
            $table->string('code')->primary(); // EF-code
            $table->string('name_en');
            $table->string('name_nl')->nullable();
            $table->string('datatype'); // code | numeric | logical | text
            $table->string('unit_code')->nullable(); // FK naar units
            $table->text('description_en')->nullable();
            $table->text('description_nl')->nullable();
            $table->timestamps();
        });

        // 3. Values (EV)
        Schema::create('etim_values', function (Blueprint $table) {
            $table->string('code')->primary(); // EV-code
            $table->string('feature_code')->nullable(); // FK naar EF
            $table->string('label_en');
            $table->string('label_nl')->nullable();
            $table->integer('sort_index')->nullable();
            $table->timestamps();

            $table->foreign('feature_code')->references('code')->on('etim_features')->cascadeOnDelete();
        });

        // 4. Units
        Schema::create('etim_units', function (Blueprint $table) {
            $table->string('code')->primary(); // unit_code
            $table->string('symbol');
            $table->timestamps();
        });

        // 5. Koppel: Class ↔ Feature
        Schema::create('etim_class_feature', function (Blueprint $table) {
            $table->id();
            $table->string('class_code'); // EC
            $table->string('feature_code'); // EF
            $table->integer('order_index')->nullable();
            $table->boolean('is_mandatory')->default(false);
            $table->timestamps();

            $table->foreign('class_code')->references('code')->on('etim_classes')->cascadeOnDelete();
            $table->foreign('feature_code')->references('code')->on('etim_features')->cascadeOnDelete();
        });

        Schema::create('etim_class_feature_values', function (Blueprint $t) {
            $t->id();
            $t->string('class_code');   // EC
            $t->string('feature_code'); // EF
            $t->string('value_code');   // EV
            $t->integer('sort_index')->nullable();
            $t->timestamps();

            $t->unique(['class_code','feature_code','value_code'], 'etim_cfval_unique');
        });

        Schema::create('etim_class_synonyms', function (Blueprint $t) {
            $t->id();
            $t->string('class_code');   // EC
            $t->string('synonym_en');
            $t->string('synonym_nl')->nullable();
            $t->timestamps();

            $t->index(['class_code']);
        });

        Schema::table('etim_class_feature', function (Blueprint $t) {
            $t->string('feature_type', 2)->nullable(); // 'A','L',...
            $t->string('unit_code')->nullable();
        });

        Schema::create('etim_groups', function (Blueprint $table) {
            $table->string('code')->primary();   // bv. EG000001
            $table->string('name_en')->nullable(); // GROUPDESC uit CSV
            $table->string('name_nl')->nullable();
            $table->timestamps();
        });

        Schema::create('etim_feature_groups', function (Blueprint $table) {
            $table->string('code')->primary();       // bv. EFG00001
            $table->string('name_en')->nullable();  // FEATUREGROUPDESC uit CSV
            $table->string('name_nl')->nullable();
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('etim_class_feature');
        Schema::dropIfExists('etim_units');
        Schema::dropIfExists('etim_values');
        Schema::dropIfExists('etim_features');
        Schema::dropIfExists('etim_classes');
        Schema::dropIfExists('etim_groups');
        Schema::dropIfExists('etim_class_feature_values');
        Schema::dropIfExists('etim_class_synonyms');
        Schema::dropIfExists('etim_feature_groups');
    }
};
