<?php

namespace Daylight\Connector2BA\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use League\Csv\Reader;

// Models
use Daylight\Connector2BA\Models\EtimClass;
use Daylight\Connector2BA\Models\EtimFeature;
use Daylight\Connector2BA\Models\EtimValue;
use Daylight\Connector2BA\Models\EtimUnit;
use Daylight\Connector2BA\Models\EtimClassFeature;
use Daylight\Connector2BA\Models\EtimClassSynonyms;
use Daylight\Connector2BA\Models\EtimFeatureGroup;
use Daylight\Connector2BA\Models\EtimGroup;

/**
 * Importeer ETIM-10 CSV datasets (classes, features, values, units, mappings)
 * Source: https://www.etim-international.com/downloads/?_sft_version=10-0&_sft_language=etim-english&_sft_unit=metric
 */

class ImportEtimData extends Command
{
    protected $signature = 'etim:import
                            {path=storage/app/private/etim : Directory met ETIM CSV\'s}
                            {--dry-run : Parse alleen, schrijf niet weg}';

    protected $description = 'Importeer ETIM-10 CSV datasets (classes, features, values, units, mappings)';

    private array $cfIndex = [];

    public function handle(): int
    {
        $dir = base_path($this->argument('path'));
        $dry = (bool) $this->option('dry-run');

        $files = [
            'ETIMUNIT.csv',
            'ETIMFEATURE.csv',
            'ETIMARTCLASS.csv',
            'ETIMARTCLASSFEATUREMAP.csv',
            'ETIMVALUE.csv',
            'ETIMARTCLASSFEATUREVALUEMAP.csv',
            'ETIMARTGROUP.csv',
            'ETIMFEATUREGROUP.csv',
            'ETIMARTCLASSSYNONYMMAP.csv',
        ];

        foreach ($files as $f) {
            if (!is_file($dir . DIRECTORY_SEPARATOR . $f)) {
                $this->warn("Bestand niet gevonden: {$f} (skip)");
            }
        }

        $this->info('Start import…' . ($dry ? ' (dry-run)' : ''));

        // 1) Units
        $this->task('Units', function () use ($dir, $dry) {
            $this->importUnits($dir.'/ETIMUNIT.csv', $dry);
        });

        // 2) Features (let op: headers/delimiter/encoding)
        $this->task('Features', function () use ($dir, $dry) {
            $this->importFeatures($dir.'/ETIMFEATURE.csv', $dry);
        });

        // 3) Classes
        $this->task('Classes', function () use ($dir, $dry) {
            $this->importClasses($dir.'/ETIMARTCLASS.csv', $dry);
        });

        // 4) Class ↔ Feature map (vult ook $cfIndex)
        $this->task('ClassFeature map', function () use ($dir, $dry) {
            $this->importClassFeatures($dir.'/ETIMARTCLASSFEATUREMAP.csv', $dry);
        });

        // 5) Values (globale lijst)
        $this->task('Values', function () use ($dir, $dry) {
            $this->importValues($dir.'/ETIMVALUE.csv', $dry);
        });

        // 6) Toegestane values per class+feature
        $this->task('ClassFeatureValue map', function () use ($dir, $dry) {
            $this->importClassFeatureValues($dir.'/ETIMARTCLASSFEATUREVALUEMAP.csv', $dry);
        });

        // 7) Groups (optioneel)
        $this->task('Groups (optioneel)', function () use ($dir, $dry) {
            $this->importGroups($dir.'/ETIMARTGROUP.csv', $dry);
        });

        // 8) FeatureGroups (optioneel)
        $this->task('FeatureGroups (optioneel)', function () use ($dir, $dry) {
            $this->importFeatureGroups($dir.'/ETIMFEATUREGROUP.csv', $dry);
        });

        // 9) Class synoniemen (optioneel)
        $this->task('ClassSynonyms (optioneel)', function () use ($dir, $dry) {
            $this->importClassSynonyms($dir.'/ETIMARTCLASSSYNONYMMAP.csv', $dry);
        });

        $this->info('Klaar!');
        return self::SUCCESS;
    }

    /* ---------------------- Helpers ---------------------- */

    /** Robuuste CSV reader: converteert UTF-16LE → UTF-8, autodetect delimiter ( ; of \t ), zet header op rij 0 */
    private function readCsv(string $file): Reader
    {
        if (!is_file($file)) {
            // maak lege reader voor consistentie
            $tmp = tmpfile();
            $meta = stream_get_meta_data($tmp);
            $csv = Reader::createFromPath($meta['uri'], 'r');
            $csv->setHeaderOffset(0);
            return $csv;
        }

        $raw = file_get_contents($file) ?: '';

        // Probeer te detecteren of het UTF-16LE is (komt vaak voor bij ETIM)
        // Heuristiek: veel \x00 bytes → behandel als UTF-16LE
        $isUtf16 = substr_count(substr($raw, 0, 2000), "\x00") > 10;
        $utf8 = $isUtf16 ? mb_convert_encoding($raw, 'UTF-8', 'UTF-16LE') : $raw;

        // Strip BOM indien aanwezig
        $utf8 = preg_replace('/^\xEF\xBB\xBF/', '', $utf8);

        // Tijdelijk naar file
        $tmp = tmpfile();
        fwrite($tmp, $utf8);
        $meta = stream_get_meta_data($tmp);

        $csv = Reader::createFromPath($meta['uri'], 'r');

        // Delimiter: eerst puntkomma proberen, anders tab als alles in 1 kolom valt
        $csv->setDelimiter(';');
        $csv->setHeaderOffset(0);
        $headers = iterator_to_array($csv->getHeader());
        if (count($headers) === 1) {
            // opnieuw openen met tab
            $csv = Reader::createFromPath($meta['uri'], 'r');
            $csv->setDelimiter("\t");
            $csv->setHeaderOffset(0);
        }

        return $csv;
    }

    private function task(string $label, \Closure $fn): void
    {
        $this->line("• {$label} …");
        $fn();
    }

    /* ---------------------- Imports per file ---------------------- */

    private function importUnits(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Verwacht: UNITID, UNITSYMBOL, UNITDESC
            $payload = [
                'code'    => $row['UNITOFMEASID'] ?? null,
                'symbol'  => $row['UNITDESC']   ?? null,
                'name_en' => $row['UNITDESC']     ?? null,
            ];
            if (!$payload['code']) { continue; }

            if (!$dry) {
                EtimUnit::updateOrCreate(['code' => $payload['code']], [
                    'symbol'  => $payload['symbol'],
                    'name_en' => $payload['name_en'],
                ]);
            }
            $count++;
        }
        $this->info("   ↳ {$count} units");
    }

    private function importFeatures(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Verwacht: FEATUREID, FEATUREDESC, FEATURETYPE, UNITOFMEASID, FEATUREGROUPID
            // In jouw screenshots leek delimiter/encoding eerst mis; met readCsv() zou dit nu goed moeten gaan.
            $featureId = $row['FEATUREID'] ?? null;
            if (!$featureId) { continue; }

            $typeChar = $row['FEATURETYPE'] ?? null;
            $unitId   = $row['UNITOFMEASID'] ?? null;

            // Vuistregel voor datatype
            $datatype =
                ($typeChar === 'L') ? 'logical' :
                    (($unitId) ? 'numeric' : 'code');

            if (!$dry) {
                EtimFeature::updateOrCreate(['code' => $featureId], [
                    'name_en'   => $row['FEATUREDESC'] ?? null,
                    'datatype'  => $datatype,
                    'unit_code' => $unitId ?: null,
                    // 'description_en' ontbreekt in deze dump
                ]);
            }
            $count++;
        }
        $this->info("   ↳ {$count} features");
    }

    private function importClasses(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: ARTCLASSID, ARTGROUPID, ARTCLASSDESC, ARTCLASSVERSION, ARTCLASSVERSIONDATE
            $code = $row['ARTCLASSID'] ?? null;
            if (!$code) { continue; }

            if (!$dry) {
                EtimClass::updateOrCreate(['code' => $code], [
                    'sector'  => $row['ARTGROUPID'] ?? null,
                    'name_en' => $row['ARTCLASSDESC'] ?? null,
                    'version' => ($row['ARTCLASSVERSION'] ?? null) ?: '10.0',
                ]);
            }
            $count++;
        }
        $this->info("   ↳ {$count} classes");
    }

    private function importClassFeatures(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: ARTCLASSFEATURENR, ARTCLASSID, FEATUREID, FEATURETYPE, UNITOFMEASID, SORTNR
            $class = $row['ARTCLASSID'] ?? null;
            $feat  = $row['FEATUREID'] ?? null;
            $mapId = $row['ARTCLASSFEATURENR'] ?? null;
            if (!$class || !$feat) { continue; }

            if (!$dry) {
                EtimClassFeature::updateOrCreate(
                    ['class_code' => $class, 'feature_code' => $feat],
                    [
                        'order_index'  => isset($row['SORTNR']) ? (int)$row['SORTNR'] : null,
                        // Als je deze kolommen aan je pivot toevoegt, kun je ze hier bewaren:
                        'feature_type' => isset($row['FEATURETYPE']) ? $row['FEATURETYPE'] : null,
                        'unit_code'    => isset($row['UNITOFMEASID']) ? $row['UNITOFMEASID'] : null,
                        'is_mandatory' => false,
                    ]
                );
            }

            if ($mapId) {
                $this->cfIndex[$mapId] = ['class_code' => $class, 'feature_code' => $feat];
            }
            $count++;
        }
        $this->info("   ↳ {$count} class-feature koppelingen");
    }

    private function importValues(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: VALUEID, VALUEDESC  (geen FEATUREID in jouw dump)
            $code = $row['VALUEID'] ?? null;
            if (!$code) { continue; }

            if (!$dry) {
                EtimValue::updateOrCreate(['code' => $code], [
                    'label_en' => $row['VALUEDESC'] ?? null,
                    // feature_code ontbreekt → laat null
                ]);
            }
            $count++;
        }
        $this->info("   ↳ {$count} values");
    }

    private function importClassFeatureValues(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: ARTCLASSFEATUREVALUENR, ARTCLASSFEATURENR, VALUEID, SORTNR
            $mapId = $row['ARTCLASSFEATURENR'] ?? null;
            $val   = $row['VALUEID'] ?? null;
            if (!$mapId || !$val) { continue; }

            if (!isset($this->cfIndex[$mapId])) {
                $this->warn("   ↳ mapping ontbreekt voor ARTCLASSFEATURENR={$mapId} (skip)");
                continue;
            }
            $cf = $this->cfIndex[$mapId];

            if (!$dry) {
                DB::table('etim_class_feature_values')->updateOrInsert(
                    [
                        'class_code'   => $cf['class_code'],
                        'feature_code' => $cf['feature_code'],
                        'value_code'   => $val,
                    ],
                    [
                        'sort_index' => isset($row['SORTNR']) ? (int)$row['SORTNR'] : null,
                        'updated_at' => now(),
                        'created_at' => now(),
                    ]
                );
            }
            $count++;
        }
        $this->info("   ↳ {$count} class-feature-value koppelingen");
    }

    private function importGroups(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: ARTGROUPID, GROUPDESC
            $code = $row['ARTGROUPID'] ?? null;
            if (!$code) { continue; }

            if (!$dry) {
                EtimGroup::updateOrInsert(
                    ['code' => $code],
                    [
                        'name_en'   => $row['GROUPDESC'] ?? null,
                        'updated_at'=> now(),
                        'created_at'=> now(),
                    ]
                );
            }
            $count++;
        }
        $this->info("   ↳ {$count} groups");
    }

    private function importFeatureGroups(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: FEATUREGROUPID, FEATUREGROUPDESC
            $code = $row['FEATUREGROUPID'] ?? null;
            if (!$code) { continue; }

            if (!$dry) {
                EtimFeatureGroup::updateOrInsert(
                    ['code' => $code],
                    [
                        'name_en'   => $row['FEATUREGROUPDESC'] ?? null,
                        'updated_at'=> now(),
                        'created_at'=> now(),
                    ]
                );
            }
            $count++;
        }
        $this->info("   ↳ {$count} feature groups");
    }

    private function importClassSynonyms(string $file, bool $dry): void
    {
        $csv = $this->readCsv($file);
        $count = 0;

        foreach ($csv as $row) {
            // Headers: ARTCLASSID, CLASSSYNONYM
            $class  = $row['ARTCLASSID'] ?? null;
            $syn    = $row['CLASSSYNONYM'] ?? null;
            if (!$class || !$syn) { continue; }

            if (!$dry) {
                EtimClassSynonyms::updateOrInsert(
                    ['class_code' => $class, 'synonym_en' => $syn],
                    [
                        'updated_at' => now(),
                        'created_at' => now()
                    ]
                );
            }
            $count++;
        }
        $this->info("   ↳ {$count} class synonyms");
    }
}
