<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector2BA\Jobs\UpdateVariant;
use Daylight\Connector2BA\Models\DataItem;
use Daylight\Connector2BA\Models\EtimClass;
use Daylight\Connector2BA\Models\EtimClassFeature;
use Daylight\Connector2BA\Models\EtimFeature;
use Daylight\Connector2BA\Models\EtimFeatureGroup;
use Daylight\Connector2BA\Models\EtimGroup;
use Daylight\Connector2BA\Models\EtimValue;
use Daylight\Connector2BA\Models\EtimUnit;
use Daylight\ConnectorCash\Jobs\UpdateProduct;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\LazyCollection;
use App\Jobs\SyncMeilisearchSettingsJob;
use Illuminate\Support\Str;

class ProcessEtimData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'connector:process-etim-data-2ba {--gtin= : Filter by specific GTIN}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process ETIM data from 2ba_data table and convert codes to readable format';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $iteration = 1;
        $limit = 10;

        if ($this->option('gtin')) {
            $limit = 1;
        }

        $this->info('Starting ETIM data processing...');

        LazyCollection::make(function () use (&$iteration, $limit) {
            while (true) {

                try {
                    $results = $this->getItems($iteration, $limit, $this->option('gtin'));
                } catch (\Exception $e) {
                    $this->error('Exception ' . $e->getMessage());
                    $results = collect();
                }

                yield $results;

                if ($iteration == 1 && $limit == 1) {
                    break;
                }
                $iteration++;

            }
        })->each(function (Collection $variants, $index) {
            $jobs = [];

            foreach ($variants as $variant) {
                $jobs[] = new UpdateVariant($variant);
            }

            Bus::batch($jobs)
                ->name('2ba:etim:iteration:' . ($index + 1))
                ->onQueue('2ba')
                ->dispatch();

            unset($jobs, $products);
            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        // Dispatch Meilisearch settings sync job
        SyncMeilisearchSettingsJob::dispatch();

        $this->line('-> SyncMeilisearchSettingsJob called');

        return Command::SUCCESS;
    }

    protected function getItems(int $iteration, int $limit, string $gtin = null): ?Collection
    {
        $query = DataItem::where('is_synced', 0)
            ->whereNotNull('etim')
            ->whereNotNull('etimClass')
            ->skip(($iteration * $limit) - $limit)
            ->limit($limit);

        // Apply GTIN filter if provided
        if ($gtin) {
            $query->where('gtin', $gtin);
            $this->info("Filtering by GTIN: {$gtin}");
        }

        $dataItems = $query->get();

        $processedData = new Collection();
        foreach ($dataItems as $dataItem) {
            $this->line("Processing item: {$dataItem->gtin} (Class: {$dataItem->etimClass})");

            try {
                $processedItem = $this->processDataItem($dataItem);
                $processedData->push($processedItem);

//                $this->info("Successfully processed: {$dataItem->gtin}");
            } catch (\Exception $e) {
                $this->error("Failed to process {$dataItem->gtin}: " . $e->getMessage());
                continue;
            }
        }

        return $processedData;
    }

    /**
     * Process a single DataItem and convert ETIM codes to readable data
     */
    private function processDataItem(DataItem $dataItem): Collection
    {
        // Get the ETIM class information
        $etimClass = EtimClass::find($dataItem->etimClass);
        $etimGroup = EtimGroup::find($etimClass->sector ?? null);

        if (!$etimClass) {
            throw new \Exception("ETIM Class {$dataItem->etimClass} not found");
        }

        $processedItem = [
            'dataitem_id' => $dataItem->id,
            'etim_class' => $dataItem->etimClass,
            'gln' => $dataItem->gln,
            'gtin' => $dataItem->gtin,
            'product_code' => $dataItem->productCode,
            'product_name' => $this->buildProductName($dataItem),
            'internal_code' => $dataItem->internalCode,
            'description' => $dataItem->description,
            'brand_name' => $dataItem->brandName,
            'brand' => $this->processBrand($dataItem->brand ?? null),
            'name_en' => $etimClass->name_en,
            'name_nl' => $etimClass->name_nl,
            'group_name_en' => $etimGroup->name_en ?? null,
            'group_name_nl' => $etimGroup->name_nl ?? null,
            'etim_features' => []
        ];

        // Process ETIM features
        $etimData = $dataItem->etim;

        if (!is_array($etimData)) {
            $etimData = json_decode($etimData, true);
        }

        foreach ($etimData as $featureCode => $valueCode) {
            $processedFeature = $this->processEtimFeature($featureCode, $valueCode, $dataItem->etimClass);

            if ($processedFeature) {
                $processedItem['etim_features'][] = $processedFeature;
            }
        }

        usort($processedItem['etim_features'], function ($a, $b) {
            return $a['order_index'] <=> $b['order_index'];
        });

        return collect($processedItem);
    }

    private function processBrand(?string $brand): ?array
    {
        if (is_null($brand)) {
            return null;
        }

        $brand = json_decode($brand, true);

        return array_filter($brand);
    }

    private function buildProductName(DataItem $dataItem): string
    {
        $productName = trim(
            strip_tags(
                str_replace('=', ' ', $dataItem->productName)
            )
        );

        $brandName = '';

        if ($dataItem->brand) {
            $brand = json_decode($dataItem->brand, true);
            $brandName = $brand['name'] ?? '';
        } else {
            $brandName = $dataItem->brandName ?? '';
        }

        $brandName = trim($brandName);

        if ($brandName === '') {
            return $productName;
        }

        if (! str_contains(mb_strtolower($productName), mb_strtolower($brandName))) {
            return trim($brandName . ' ' . $productName);
        }

        return $productName;
    }

    /**
     * Process a single ETIM feature and its value using EtimClassFeature
     */
    private function processEtimFeature(string $featureCode, string $valueCode, string $classCode): ?array
    {
        // Get the class-feature relationship from EtimClassFeature
        $classFeature = EtimClassFeature::where('class_code', $classCode)
            ->where('feature_code', $featureCode)
            ->with(['etimFeature', 'etimUnit'])
            ->first();

        if (!$classFeature) {
            $this->warn("ETIM ClassFeature relationship not found for class {$classCode} and feature {$featureCode}");
            return null;
        }

        $feature = $classFeature->etimFeature;

        if (!$feature) {
            $this->warn("ETIM Feature {$featureCode} not found");
            return null;
        }

        $processedFeature = [
            'feature_code' => $feature->code,
            'feature_name_en' => $feature->name_en,
            'feature_name_nl' => $feature->name_nl,
            //'datatype' => $feature->datatype,
            'featuretype' => $classFeature->feature_type,
            //'raw_value' => $valueCode,
            //'processed_value' => null,
            'unit_symbol' => null,
            'order_index' => $classFeature->order_index,
            //'is_mandatory' => $classFeature->is_mandatory,
            //'feature_type' => $classFeature->feature_type,
        ];

        // Add unit information - prioritize unit from class-feature relationship, then from feature
        $unitCode = $classFeature->unit_code ?? $feature->unit_code;

        if ($unitCode) {
            $unit = $classFeature->etimUnit ?? $feature->unit;
            if ($unit) {
                $processedFeature['unit_symbol'] = $unit->symbol;
            }
        }

        // Process the value based on datatype
        switch (strtoupper($classFeature->feature_type)) {
            case 'A': // Alphanumeric: Materiaal = “Staal”, “Kunststof”, “Aluminium
                // For code datatype, look up the value in etim_values
                $value = EtimValue::find($valueCode);
                if ($value && !is_null($value)) {
                    $processedFeature['processed_value'] = [
                        'label_en' => $value->label_en,
                        'label_nl' => $value->label_nl,
                    ];
                } else {
                    if (in_array($valueCode, ['MV', 'NA', 'UN'])) {
                        $processedFeature['processed_value'] = false;
                    } else {
                        // Fallback: use the value code as-is if not found
                        $processedFeature['processed_value'] = $valueCode;
                    }
                }
                break;

            case 'N': // Numeric: Gewicht = 12 kg
                // For numeric values, keep as is but add unit context
                if (in_array($valueCode, ['MV', 'NA', 'UN'])) {
                    $processedFeature['processed_value'] = false;
                } else {
                    $processedFeature['processed_value'] = (float)$valueCode;
                }
                break;

            case 'L': // Logical: Waterdicht = Ja/Nee (Y,N)
                // For logical values, convert to boolean
                if (in_array($valueCode, ['MV', 'NA', 'UN'])) {
                    $processedFeature['processed_value'] = false;
                } else {
                    $processedFeature['processed_value'] = filter_var($valueCode, FILTER_VALIDATE_BOOLEAN) ? 'Y' : 'N';
                }
                break;

            case 'R': // Range: Temperatuurbereik = -20 … +40 °C
                // For range values, keep as string (could be "10-20" etc.)
                if (in_array($valueCode, ['MV', 'NA', 'UN'])) {
                    $processedFeature['processed_value'] = false;
                } else {
                    $processedFeature['processed_value'] = str_replace('.000', '', $valueCode);
                }
                break;
            default:
                // For text or unknown types, keep as string
                if (in_array($valueCode, ['MV', 'NA', 'UN'])) {
                    $processedFeature['processed_value'] = false;
                } else {
                    $processedFeature['processed_value'] = $valueCode;
                }
                break;
        }

        // Don't process features with processed_value === false
        if ($processedFeature['processed_value'] === false) {
            return null;
        }

        return $processedFeature;
    }
}
