<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Core\Modules\MediaLibrary\Jobs\GenerateMediaConversionJob;
use Daylight\Core\Modules\MediaLibrary\Models\Media;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Laravel\Facades\Image;

class Sync2baAttachmentsToVariants extends Command
{
    protected $signature = 'connector:sync-2ba-attachments
                            {--force : Re-sync alle records, ook die al gesynct zijn}
                            {--limit= : Beperk het aantal te verwerken records}';

    protected $description = 'Synchroniseer attachments van DataItem records naar Variants';

    public function handle(): int
    {
        $force = $this->option('force');
        $limit = $this->option('limit') ? (int) $this->option('limit') : null;

        $query = DB::table('2ba_data')
            ->where(function ($q) use ($force) {
                if (! $force) {
                    $q->where('is_media_synced', false)
                        ->orWhereNull('is_media_synced');
                }
            })
            ->whereNotNull('attachments')
            ->where('attachments', '!=', '')
            ->whereRaw('JSON_LENGTH(JSON_EXTRACT(attachments, "$.images")) > 0')
            ->whereNotNull('gtin')
            ->where('gtin', '!=', '')
            ->orderBy('id');

        if ($limit) {
            $query->limit($limit);
        }

        $total = $query->count();

        if ($total === 0) {
            $this->info('Geen records gevonden om te synchroniseren.');

            return self::SUCCESS;
        }

        $this->info("Gevonden {$total} records om te synchroniseren.");
        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $processed = 0;
        $errors = 0;

        $query->chunk(100, function ($dataItems) use (&$processed, &$errors, $bar) {
            foreach ($dataItems as $dataItem) {
                try {
                    $this->processDataItem($dataItem);
                    $processed++;
                } catch (\Exception $e) {
                    $errors++;
                    Log::error('Fout bij verwerken DataItem', [
                        'id' => $dataItem->id,
                        'gtin' => $dataItem->gtin,
                        'error' => $e->getMessage(),
                    ]);
                    $this->warn("\nFout bij DataItem ID {$dataItem->id}: {$e->getMessage()}");
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);
        $this->info("Verwerking voltooid: {$processed} succesvol, {$errors} fouten.");

        return self::SUCCESS;
    }

    protected function processDataItem(object $dataItem): void
    {
        $attachments = json_decode($dataItem->attachments, true);

        if (! is_array($attachments) || ! isset($attachments['images']) || ! is_array($attachments['images'])) {
            $this->markAsSynced($dataItem->id);

            return;
        }

        if (empty($attachments['images'])) {
            $this->markAsSynced($dataItem->id);

            return;
        }

        $variant = daylightModel('variant')::where('gtin_product', $dataItem->gtin)->first();

        if (! $variant) {
            Log::warning('Geen Variant gevonden voor GTIN', [
                'gtin' => $dataItem->gtin,
                'data_item_id' => $dataItem->id,
            ]);

            return;
        }

        $variantsWithSameGtin = daylightModel('variant')::where('gtin_product', $dataItem->gtin)->count();

        if ($variantsWithSameGtin > 1) {
            Log::warning('Meerdere Variants gevonden met dezelfde GTIN, gebruik eerste', [
                'gtin' => $dataItem->gtin,
                'count' => $variantsWithSameGtin,
                'data_item_id' => $dataItem->id,
            ]);
        }

        $existingMediaCount = $variant->media()->count();
        $order = $existingMediaCount;
        $successCount = 0;

        foreach ($attachments['images'] as $image) {
            if (! isset($image['location']) || empty($image['location'])) {
                continue;
            }

            $imageUrl = $image['location'];

            if (! filter_var($imageUrl, FILTER_VALIDATE_URL)) {
                Log::warning('Ongeldige URL in attachments', [
                    'url' => $imageUrl,
                    'data_item_id' => $dataItem->id,
                ]);

                continue;
            }

            try {
                $this->addMediaToVariant($variant, $imageUrl, $order);
                $order++;
                $successCount++;
            } catch (\Exception $e) {
                Log::error('Fout bij toevoegen media aan Variant', [
                    'variant_id' => $variant->id,
                    'gtin' => $dataItem->gtin,
                    'url' => $imageUrl,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        if ($successCount > 0) {
            $this->markAsSynced($dataItem->id);
        }
    }

    protected function addMediaToVariant($variant, string $url, int $order): void
    {
        $imageContents = $this->downloadImageContents($url);
        $imageName = basename(parse_url($url, PHP_URL_PATH));
        $originalExtension = strtolower(pathinfo($imageName, PATHINFO_EXTENSION));

        // Converteer TIFF naar JPG
        $extension = $originalExtension;
        $finalImageContents = $imageContents;

        if (in_array($originalExtension, ['tiff', 'tif'])) {
            try {
                $image = Image::read($imageContents);
                $image->orient();
                $finalImageContents = (string) $image->toJpeg(100);
                $extension = 'jpg';
                $imageName = pathinfo($imageName, PATHINFO_FILENAME).'.jpg';
            } catch (\Exception $e) {
                Log::warning('Kon TIFF niet converteren naar JPG, gebruik origineel', [
                    'url' => $url,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        $storagePath = 'media/products/'.uniqid().'_'.$imageName;
        $disk = config('media.disk', 'public');

        $stored = Storage::disk($disk)->put($storagePath, $finalImageContents, 'public');

        if (! $stored) {
            throw new \Exception("Kon afbeelding niet opslaan op disk: {$disk}, path: {$storagePath}");
        }

        // Wacht tot het bestand daadwerkelijk beschikbaar is
        $maxAttempts = 10;
        $attempt = 0;
        while ($attempt < $maxAttempts) {
            if (Storage::disk($disk)->exists($storagePath)) {
                break;
            }
            usleep(250000);
            $attempt++;
        }

        if ($attempt >= $maxAttempts) {
            throw new \Exception("Bestand niet beschikbaar na upload: {$storagePath}");
        }

        $media = Media::create([
            'name' => $imageName,
            'directory_id' =>  config('services.connector.2ba.image_directory_id'),
            'size' => strlen($finalImageContents),
            'handle' => $storagePath,
            'path' => $storagePath,
            'disk' => $disk,
            'extension' => $extension,
            'type' => 'image/'.$extension,
            'hash' => hash('sha256', $finalImageContents),
        ]);

        $conversions = config('media.conversions', []);

        foreach ($conversions as $conversionName => $config) {
            GenerateMediaConversionJob::dispatch($media, $conversionName, $config);
        }

        $existingAttachment = $variant->media()
            ->where('media_id', $media->id)
            ->first();

        if ($existingAttachment) {
            $variant->media()->updateExistingPivot($media->id, [
                'order' => $order,
            ]);
        } else {
            $variant->media()->attach($media->id, [
                'order' => $order,
            ]);
        }
    }

    protected function downloadImageContents(string $url): string
    {
        try {
            $response = Http::timeout(30)->get($url);

            if (! $response->successful()) {
                throw new \Exception("Kon afbeelding niet downloaden: HTTP {$response->status()}");
            }

            return $response->body();
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            throw new \Exception("Verbindingsfout bij downloaden afbeelding: {$e->getMessage()}");
        } catch (\Exception $e) {
            throw new \Exception("Fout bij downloaden afbeelding: {$e->getMessage()}");
        }
    }

    protected function markAsSynced(int $dataItemId): void
    {
        DB::table('2ba_data')
            ->where('id', $dataItemId)
            ->update(['is_media_synced' => true]);
    }
}

