<?php

namespace Daylight\Connector2BA\Jobs;

use Daylight\Connector2BA\Service\ETIMReader;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Daylight\Connector2BA\Models\DataItem;
use JsonException;
use Illuminate\Support\Carbon;

class ProcessSelectionProfile implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        private string $selectionProfileId,
        private string $extractedPath,
        private ?string $supplierIdGln
    ) {
        $this->onConnection('redis');
        $this->onQueue('2ba');
    }

    public function handle(): void
    {
        try {
            Log::info('Processing 2BA SelectionProfile', [
                'selection_profile_id' => $this->selectionProfileId,
                'extracted_path' => $this->extractedPath,
            ]);

            // Get all extracted files
            $files = Storage::allFiles($this->extractedPath);

            Log::info('Found files in SelectionProfile', [
                'selection_profile_id' => $this->selectionProfileId,
                'file_count' => count($files),
                'files' => $files,
            ]);

            // Process each file
            foreach ($files as $file) {
                $this->processFile($file);
            }

            Log::info('SelectionProfile processing completed', [
                'selection_profile_id' => $this->selectionProfileId,
                'processed_files' => count($files),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to process 2BA SelectionProfile', [
                'selection_profile_id' => $this->selectionProfileId,
                'extracted_path' => $this->extractedPath,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    private function processFile(string $filePath): void
    {
        try {
            $fileName = basename($filePath);
            $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
            $fileSize = Storage::size($filePath);

            // Only handle ETIMxChangeExport.json
            if ($fileName === 'ETIMxChangeExport.json') {
                $this->processJsonFile($filePath, $fileName);
                return;
            }

        } catch (\Exception $e) {
            Log::error('Failed to process individual file', [
                'file_path' => $filePath,
                'error' => $e->getMessage(),
            ]);
        }
    }

    private function processJsonFile(string $filePath, string $fileName): void
    {
        Log::info('Processing JSON file', [
            'file_path' => $filePath,
            'file_name' => $fileName
        ]);

        try {
            // Get the full file path from storage
            $fullFilePath = Storage::path($filePath);
            $directoryPath = dirname($fullFilePath);

            Log::info('File processing details', [
                'filePath' => $filePath,
                'full_file_path' => $fullFilePath,
                'directory_path' => $directoryPath,
                'file_exists' => file_exists($fullFilePath)
            ]);

            // Initialize ETIMReader with directory path and filename
            $reader = app(ETIMReader::class);

            try {
                $reader->processTradeItems(
                    'local',
                    $filePath,
                    $this->supplierIdGln,
                    10,
                    function (array $batch) {
                        // Dispatch a new job for each batch instead of processing directly
                        ProcessDataBatch::dispatch($batch)
                            ->onConnection('redis')
                            ->onQueue('2ba');
                    }
                );
            } catch (\Throwable $e) {
                Log::error('ETIM import failed', [
                    'error_class' => get_class($e),
                    'message'     => $e->getMessage(),
                    'trace_top'   => collect(explode("\n", $e->getTraceAsString()))->take(10)->implode("\n"),
                    'disk'        => 'local',
                    'path'        => $filePath,
                ]);
                throw $e;
            }

        } catch (\Exception $e) {
            Log::error('Failed to process text file', [
                'file_path' => $filePath,
                'file_name' => $fileName,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            throw $e;
        }
    }

    /**
     * Process a batch of parsed records
     */
    private function processBatch(array $batch): void
    {
        try {
            Log::info('Processing batch data', [
                'batch_size' => count($batch),
//                'state' => ($this->supplierIdGln) ? 'insert' : 'update'
            ]);

            // Prepare data for bulk operations
            $dataToInsert = [];
            $dataToUpdate = [];

            foreach ($batch as $record) {
                $processedData = [
                    'internalCode' => $record['internalCode'],
                    'productCode' => $record['manufacturerProductNumber'],
                    'gln' => $record['gln'],
                    'gtin' => $record['gtin'],
                    'brandName' => $record['brandName'],
                    'description' => $record['productDescription'],
                    'etimClass' => $record['etimClass'],
                    'etim' => json_encode($record['etim']),
                    'attachments' => json_encode($record['attachments']),
                    'updated_at' => now(),
                ];

                // Check if record exists
                $existingRecord = DataItem::where('gtin', $record['gtin'])
                    ->first();

                if ($existingRecord) {
                    $processedData['id'] = $existingRecord->id;
                    $dataToUpdate[] = $processedData;
                } else {
                    $processedData['created_at'] = Carbon::now();
                    $dataToInsert[] = $processedData;
                }
            }

            // Perform bulk operations
            if (!empty($dataToInsert)) {
                DataItem::insert($dataToInsert);
                Log::info('Bulk inserted records', ['count' => count($dataToInsert)]);
            }

            if (!empty($dataToUpdate)) {
                foreach ($dataToUpdate as $updateData) {
                    $id = $updateData['id'];
                    unset($updateData['id']);
                    unset($updateData['internalCode']); //update only called with productdata, do not overwrite internalCode cause it is the owners code
                    DataItem::where('id', $id)->update($updateData);
                }
                Log::info('Bulk updated records', ['count' => count($dataToUpdate)]);
            }

        } catch (\Throwable $e) {
            $pdoInfo = method_exists($e, 'errorInfo') && is_array($e->errorInfo ?? null) ? $e->errorInfo : null;

            Log::error('Failed to process batch', [
                'batch_size'  => count($batch),
                'error'       => $e->getMessage(),
                'code'        => $e->getCode(),
                'exception'   => get_class($e),
                'file'        => $e->getFile(),
                'line'        => $e->getLine(),
                'trace'       => $e->getTraceAsString(),
                'pdo_error'   => $pdoInfo ? ($pdoInfo[2] ?? json_encode($pdoInfo)) : null,
            ]);

            throw $e;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('2BA SelectionProfile processing job failed permanently', [
            'selection_profile_id' => $this->selectionProfileId,
            'extracted_path' => $this->extractedPath,
            'error' => $exception->getMessage(),
        ]);
    }

    private function toValidJsonOrNull($value): ?string
    {
        if (is_string($value)) {
            json_decode($value, true);
            return (json_last_error() === JSON_ERROR_NONE) ? $value : null;
        }

        if (is_array($value) || is_object($value)) {
            try {
                $json = json_encode($value, JSON_THROW_ON_ERROR);
                return ($json === '' || $json === false) ? null : $json;
            } catch (JsonException $e) {
                return null;
            }
        }

        return null;
    }

}
