<?php

namespace Daylight\Connector2BA\Mappers;

use Daylight\Connector\Contracts\MapperInterface;
use Daylight\Connector2BA\Entities\Product as ProductEntity;

class ProductMapper implements MapperInterface
{
    /**
     * Map the data to the format required by the connector.
     *
     * @param array $data
     * @return ProductEntity
     */
    public function map(array $data): ProductEntity
    {
        return new ProductEntity(
            id: (string)($data['id'] ?? ''),
            sku: (string)($data['sku'] ?? $data['code'] ?? ''),
            ean: $data['ean'] ?? $data['barcode'] ?? null,
            name: (string)($data['name'] ?? $data['title'] ?? ''),
            shortDescription: $data['short_description'] ?? $data['summary'] ?? null,
            longDescription: $data['long_description'] ?? $data['description'] ?? null,
            marketingText: $data['marketing_text'] ?? $data['marketing_description'] ?? null,
            technicalDescription: $data['technical_description'] ?? $data['technical_specs'] ?? null,
            category: $data['category'] ?? $data['category_name'] ?? null,
            brand: $data['brand'] ?? $data['manufacturer'] ?? null,
            model: $data['model'] ?? $data['model_number'] ?? null,
            specifications: $this->mapSpecifications($data),
            features: $data['features'] ?? null,
            attributes: $data['attributes'] ?? null,
            media: $data['media'] ?? null,
            images: $this->mapImages($data),
            documents: $this->mapDocuments($data),
            videos: $this->mapVideos($data),
            color: $data['color'] ?? $data['colour'] ?? null,
            size: $data['size'] ?? null,
            material: $data['material'] ?? null,
            dimensions: $this->mapDimensions($data),
            weight: isset($data['weight']) ? (float)$data['weight'] : null,
            active: $this->mapActive($data),
            createdAt: $data['created_at'] ?? $data['created'] ?? null,
            updatedAt: $data['updated_at'] ?? $data['updated'] ?? null,
        );
    }

    public function mapWithAttributes(array $data, array $attributes = []): ProductEntity
    {
        $product = $this->map($data);
        
        foreach ($attributes as $key => $value) {
            if (property_exists($product, $key)) {
                $product->{$key} = $value;
            }
        }
        
        return $product;
    }

    private function mapActive(array $data): bool
    {
        if (isset($data['active'])) {
            return (bool)$data['active'];
        }

        if (isset($data['status'])) {
            return strtolower($data['status']) === 'active';
        }

        // Default to active if no status information is available
        return true;
    }

    private function mapSpecifications(array $data): ?array
    {
        // Try different possible specification field names
        $specs = $data['specifications'] ?? $data['specs'] ?? $data['technical_data'] ?? null;
        
        if (!$specs || !is_array($specs)) {
            return null;
        }

        return $specs;
    }

    private function mapImages(array $data): ?array
    {
        $images = $data['images'] ?? $data['pictures'] ?? $data['photos'] ?? null;
        
        if (!$images || !is_array($images)) {
            return null;
        }

        // Normalize image structure
        return array_map(function ($image) {
            if (is_string($image)) {
                return ['url' => $image, 'is_main' => false];
            }
            
            return [
                'url' => $image['url'] ?? $image['src'] ?? $image['path'] ?? null,
                'alt' => $image['alt'] ?? $image['title'] ?? null,
                'is_main' => $image['is_main'] ?? $image['primary'] ?? false,
                'sort_order' => $image['sort_order'] ?? $image['order'] ?? 0,
            ];
        }, $images);
    }

    private function mapDocuments(array $data): ?array
    {
        $documents = $data['documents'] ?? $data['files'] ?? $data['attachments'] ?? null;
        
        if (!$documents || !is_array($documents)) {
            return null;
        }

        // Normalize document structure
        return array_map(function ($document) {
            if (is_string($document)) {
                return ['url' => $document, 'type' => 'unknown'];
            }
            
            return [
                'url' => $document['url'] ?? $document['path'] ?? null,
                'title' => $document['title'] ?? $document['name'] ?? null,
                'type' => $document['type'] ?? $document['mime_type'] ?? 'unknown',
                'size' => $document['size'] ?? null,
            ];
        }, $documents);
    }

    private function mapVideos(array $data): ?array
    {
        $videos = $data['videos'] ?? null;
        
        if (!$videos || !is_array($videos)) {
            return null;
        }

        // Normalize video structure
        return array_map(function ($video) {
            if (is_string($video)) {
                return ['url' => $video, 'type' => 'unknown'];
            }
            
            return [
                'url' => $video['url'] ?? $video['src'] ?? null,
                'title' => $video['title'] ?? $video['name'] ?? null,
                'type' => $video['type'] ?? 'video',
                'thumbnail' => $video['thumbnail'] ?? $video['preview'] ?? null,
                'duration' => $video['duration'] ?? null,
            ];
        }, $videos);
    }

    private function mapDimensions(array $data): ?array
    {
        // Check if dimensions are provided as separate fields
        if (isset($data['length']) || isset($data['width']) || isset($data['height'])) {
            return [
                'length' => $data['length'] ?? null,
                'width' => $data['width'] ?? null,
                'height' => $data['height'] ?? null,
                'unit' => $data['dimension_unit'] ?? 'cm',
            ];
        }

        // Check if dimensions are provided as an object
        $dimensions = $data['dimensions'] ?? null;
        if ($dimensions && is_array($dimensions)) {
            return $dimensions;
        }

        return null;
    }
}
