<?php

namespace Daylight\Connector2BA\Service;

use Illuminate\Support\LazyCollection;
use League\Csv\Reader;
use RuntimeException;

class AttachmentCSVReader
{
    /**
     * Read CSV file and return a LazyCollection of attachment data.
     *
     * @param string $filePath Full path to the CSV file
     * @return LazyCollection Yields arrays with keys: productCode, location, mutationDate, type
     */
    public function readFile(string $filePath): LazyCollection
    {
        if (!is_file($filePath) || !is_readable($filePath)) {
            throw new RuntimeException("Bestand niet leesbaar of bestaat niet: {$filePath}");
        }

        return LazyCollection::make(function () use ($filePath) {
            $csv = $this->createReader($filePath);

            foreach ($csv as $row) {
                $productCode = $row['ProductCode'] ?? null;
                $location = $row['AttachmentLocationNew'] ?? null;
                $mutationDate = $row['Mutationdate'] ?? null;

                // Skip rows without required data
                if (empty($productCode) || empty($location)) {
                    continue;
                }

                $type = $this->getAttachmentType($location);

                yield [
                    'productCode' => $productCode,
                    'location' => $location,
                    'mutationDate' => $mutationDate,
                    'type' => $type,
                ];
            }
        });
    }

    /**
     * Determine attachment type based on file extension or mimetype.
     *
     * @param string $location File location/URL
     * @return string 'image' or 'pdf'
     */
    public function getAttachmentType(string $location): string
    {
        // Extract file extension from URL/path
        $path = parse_url($location, PHP_URL_PATH) ?? $location;
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

        // Image extensions
        $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'tiff', 'tif'];

        if (in_array($extension, $imageExtensions)) {
            return 'image';
        }

        // PDF extension
        if ($extension === 'pdf') {
            return 'pdf';
        }

        // Fallback: try to detect from mimetype if it's in the URL
        // Some URLs might have mimetype hints
        if (stripos($location, 'image/') !== false) {
            return 'image';
        }

        if (stripos($location, 'application/pdf') !== false || stripos($location, '.pdf') !== false) {
            return 'pdf';
        }

        // Default to image if uncertain (most attachments are images)
        return 'image';
    }

    /**
     * Create a CSV Reader instance with proper encoding and delimiter detection.
     *
     * @param string $filePath Full path to the CSV file
     * @return Reader
     */
    private function createReader(string $filePath): Reader
    {
        $raw = file_get_contents($filePath) ?: '';

        // Detect UTF-16LE encoding (common in some CSV files)
        $isUtf16 = substr_count(substr($raw, 0, 2000), "\x00") > 10;
        $utf8 = $isUtf16 ? mb_convert_encoding($raw, 'UTF-8', 'UTF-16LE') : $raw;

        // Strip BOM if present
        $utf8 = preg_replace('/^\xEF\xBB\xBF/', '', $utf8);

        // Write to temporary file for League\Csv
        $tmp = tmpfile();
        fwrite($tmp, $utf8);
        $meta = stream_get_meta_data($tmp);

        $csv = Reader::createFromPath($meta['uri'], 'r');

        // Try semicolon delimiter first (common in European CSV files)
        $csv->setDelimiter(';');
        $csv->setHeaderOffset(0);
        $headers = iterator_to_array($csv->getHeader());

        // If only one column, try comma delimiter
        if (count($headers) === 1) {
            $csv = Reader::createFromPath($meta['uri'], 'r');
            $csv->setDelimiter(',');
            $csv->setHeaderOffset(0);
        }

        return $csv;
    }
}


