<?php

require_once __DIR__ . '/vendor/autoload.php';

use Daylight\Connector2BA\Service\ETIMReader;

// Function to get current memory usage in MB
function getMemoryUsage() {
    return round(memory_get_usage(true) / 1024 / 1024, 2);
}

function getPeakMemoryUsage() {
    return round(memory_get_peak_usage(true) / 1024 / 1024, 2);
}

// Create a large ETIM test file
function createLargeETIMFile($filePath, $supplierCount = 2, $productsPerSupplier = 100, $tradeItemsPerProduct = 5) {
    $etimData = [
        "CatalogueId" => "2BA-EXP-LARGE-TEST",
        "CatalogueName" => [
            [
                "Language" => "nl-NL",
                "CatalogueName" => "Large ETIM Test Export"
            ]
        ],
        "CatalogueVersion" => "1",
        "CatalogueType" => "FULL",
        "GenerationDate" => "2025-09-19",
        "CatalogueValidityStart" => "2025-09-19",
        "Language" => ["nl-NL"],
        "Supplier" => []
    ];

    for ($s = 0; $s < $supplierCount; $s++) {
        $supplier = [
            "SupplierName" => "Test Supplier " . ($s + 1),
            "SupplierIdGln" => "871425303888" . $s,
            "Product" => []
        ];

        for ($p = 0; $p < $productsPerSupplier; $p++) {
            $productId = ($s * $productsPerSupplier) + $p;
            $product = [
                "ProductIdentification" => [
                    "ManufacturerIdGln" => "004218600001" . ($productId % 10),
                    "ManufacturerName" => "Test Manufacturer " . ($productId % 20),
                    "ManufacturerProductNumber" => "PROD" . str_pad($productId, 6, '0', STR_PAD_LEFT),
                    "ProductGtin" => ["0421866872" . str_pad($productId, 4, '0', STR_PAD_LEFT)]
                ],
                "TradeItem" => []
            ];

            for ($t = 0; $t < $tradeItemsPerProduct; $t++) {
                $tradeItemId = ($productId * $tradeItemsPerProduct) + $t;
                $tradeItem = [
                    "ItemIdentification" => [
                        "SupplierItemNumber" => "TI" . str_pad($tradeItemId, 8, '0', STR_PAD_LEFT),
                        "ItemGtin" => ["0421866872" . str_pad($tradeItemId, 4, '0', STR_PAD_LEFT)],
                        "DiscountGroupId" => "60171"
                    ],
                    "ItemDetails" => [
                        "ItemStatus" => ($tradeItemId % 3 === 0) ? "DISCONTINUED" : "ACTIVE",
                        "StockItem" => ($tradeItemId % 2 === 0),
                        "ItemDescriptions" => [
                            [
                                "DescriptionLanguage" => "nl-NL",
                                "MinimalItemDescription" => "Test Product " . $tradeItemId . " - " . str_repeat("Lorem ipsum dolor sit amet ", 20)
                            ]
                        ]
                    ],
                    "ItemLogisticDetails" => [[]],
                    "Ordering" => [
                        "OrderUnit" => "C62",
                        "MinimumOrderQuantity" => (string)(1 + ($tradeItemId % 10)) . ".000",
                        "OrderStepSize" => (string)(1 + ($tradeItemId % 5)) . ".000",
                        "UseUnit" => "C62",
                        "UseUnitConversionFactor" => "1.000",
                        "SingleUseUnitQuantity" => "1.000"
                    ],
                    "Pricing" => [
                        [
                            "PriceUnit" => "C62",
                            "PriceUnitFactor" => "1.000",
                            "PriceQuantity" => "1.000",
                            "PriceOnRequest" => false,
                            "GrossListPrice" => (string)(50 + ($tradeItemId * 0.5)) . ".000",
                            "Vat" => "21.00",
                            "PriceValidityDate" => "2023-08-02"
                        ]
                    ],
                    "ItemCountrySpecificFields" => [
                        [
                            "CSItemCharacteristicCode" => "NL_012",
                            "CSItemCharacteristicName" => [
                                [
                                    "CSItemCharacteristicName" => "Orderable indicator"
                                ]
                            ],
                            "CSItemCharacteristicValueBoolean" => ($tradeItemId % 4 !== 0)
                        ]
                    ]
                ];

                $product["TradeItem"][] = $tradeItem;
            }

            $supplier["Product"][] = $product;
        }

        $etimData["Supplier"][] = $supplier;
    }

    file_put_contents($filePath, json_encode($etimData, JSON_PRETTY_PRINT));
    
    return $supplierCount * $productsPerSupplier * $tradeItemsPerProduct;
}

// Create test directory and file
$testDir = __DIR__ . '/test_data';
if (!is_dir($testDir)) {
    mkdir($testDir, 0755, true);
}

$testFile = $testDir . '/LargeETIMxChangeExport.json';

echo "Creating large ETIM test file...\n";
echo "Memory before file creation: " . getMemoryUsage() . " MB\n";

$expectedItems = createLargeETIMFile($testFile, 3, 200, 10); // 3 suppliers, 200 products each, 10 trade items each = 6000 items

$fileSize = round(filesize($testFile) / 1024 / 1024, 2);
echo "Test file created: $testFile ($fileSize MB)\n";
echo "Expected TradeItems: $expectedItems\n";
echo "Memory after file creation: " . getMemoryUsage() . " MB\n\n";

// Test the memory-optimized ETIMReader
try {
    echo "Testing memory-optimized ETIM ETIMReader:\n";
    echo "========================================\n";
    
    $memoryBefore = getMemoryUsage();
    echo "Memory before processing: $memoryBefore MB\n";
    
    $reader = new ETIMReader($testDir, 'LargeETIMxChangeExport.json');
    
    $batchNumber = 1;
    $totalProcessed = 0;
    $maxMemoryDuringProcessing = $memoryBefore;
    
    foreach ($reader->readFile(50) as $batch) {
        $currentMemory = getMemoryUsage();
        $maxMemoryDuringProcessing = max($maxMemoryDuringProcessing, $currentMemory);
        
        echo "Batch $batchNumber: " . count($batch) . " items, Memory: {$currentMemory} MB\n";
        
        // Verify first item in first batch
        if ($batchNumber === 1 && !empty($batch)) {
            $firstItem = $batch[0];
            echo "  First item TradeItemId: " . ($firstItem['TradeItemId'] ?? 'N/A') . "\n";
            echo "  First item GTIN: " . ($firstItem['GTIN'] ?? 'N/A') . "\n";
            echo "  First item Supplier: " . ($firstItem['SupplierName'] ?? 'N/A') . "\n";
        }
        
        $totalProcessed += count($batch);
        $batchNumber++;
        
        // Show progress every 20 batches
        if ($batchNumber % 20 === 0) {
            echo "  Progress: $totalProcessed items processed\n";
        }
    }
    
    $memoryAfter = getMemoryUsage();
    $peakMemory = getPeakMemoryUsage();
    
    echo "\nProcessing completed!\n";
    echo "Total items processed: $totalProcessed\n";
    echo "Expected items: $expectedItems\n";
    echo "Memory before: $memoryBefore MB\n";
    echo "Memory after: $memoryAfter MB\n";
    echo "Peak memory usage: $peakMemory MB\n";
    echo "Max memory during processing: $maxMemoryDuringProcessing MB\n";
    echo "Memory increase: " . round($maxMemoryDuringProcessing - $memoryBefore, 2) . " MB\n";
    
    // Verify results
    if ($totalProcessed === $expectedItems) {
        echo "✅ STRUCTURE SUCCESS: Correctly processed all $expectedItems TradeItems\n";
    } else {
        echo "❌ STRUCTURE ERROR: Expected $expectedItems items, got $totalProcessed\n";
    }
    
    if ($maxMemoryDuringProcessing - $memoryBefore < 100) {
        echo "✅ MEMORY SUCCESS: Low memory usage increase\n";
    } else {
        echo "⚠️  MEMORY WARNING: Memory usage higher than expected\n";
    }
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

// Clean up test file
unlink($testFile);
rmdir($testDir);
echo "\nTest completed and cleaned up.\n";

