<?php

namespace Daylight\Connector2BA\Jobs;

use Daylight\Connector2BA\Models\DataItem;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use JsonException;

class ProcessDataBatch implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        private array $batch
    ) {
        $this->onConnection('redis');
        $this->onQueue('2ba');
    }

    public function handle(): void
    {
        try {
            Log::info('Processing batch data', [
                'batch_size' => count($this->batch),
            ]);

            // Prepare data for bulk operations
            $dataToInsert = [];
            $dataToUpdate = [];

            foreach ($this->batch as $record) {
                $processedData = [
                    'internalCode' => $record['internalCode'],
                    'productCode' => $record['manufacturerProductNumber'],
                    'gln' => $record['gln'],
                    'gtin' => $record['gtin'],
                    'brandName' => $record['brandName'],
                    'description' => $record['productDescription'],
                    'etimClass' => $record['etimClass'],
                    'etim' => json_encode($record['etim']),
                    'attachments' => json_encode($record['attachments']),
                    'updated_at' => now(),
                ];

                // Check if record exists
                $existingRecord = DataItem::where('gtin', $record['gtin'])
                    ->first();

                if ($existingRecord) {
                    $processedData['id'] = $existingRecord->id;
                    $dataToUpdate[] = $processedData;
                } else {
                    $processedData['created_at'] = Carbon::now();
                    $dataToInsert[] = $processedData;
                }
            }

            // Perform bulk operations
            if (!empty($dataToInsert)) {
                DataItem::insert($dataToInsert);
                Log::info('Bulk inserted records', ['count' => count($dataToInsert)]);
            }

            if (!empty($dataToUpdate)) {
                foreach ($dataToUpdate as $updateData) {
                    $id = $updateData['id'];
                    unset($updateData['id']);
                    unset($updateData['internalCode']); //update only called with productdata, do not overwrite internalCode cause it is the owners code
                    DataItem::where('id', $id)->update($updateData);
                }
                Log::info('Bulk updated records', ['count' => count($dataToUpdate)]);
            }

        } catch (\Throwable $e) {
            $pdoInfo = method_exists($e, 'errorInfo') && is_array($e->errorInfo ?? null) ? $e->errorInfo : null;

            Log::error('Failed to process batch', [
                'batch_size'  => count($this->batch),
                'error'       => $e->getMessage(),
                'code'        => $e->getCode(),
                'exception'   => get_class($e),
                'file'        => $e->getFile(),
                'line'        => $e->getLine(),
                'trace'       => $e->getTraceAsString(),
                'pdo_error'   => $pdoInfo ? ($pdoInfo[2] ?? json_encode($pdoInfo)) : null,
            ]);

            throw $e;
        }
    }
}
