<?php

namespace Daylight\Connector2BA\Console;

use Daylight\Connector2BA\Http\TwoBA;
use Illuminate\Console\Command;

/**
 * Command to request direct profile from 2BA API
 * This command uses the RequestDirectProfile endpoint to get product information
 * for specific supplier GLN and trade item IDs.
 *
 * @examples
 * - php artisan connector:request-direct-profile-2ba
 * - php artisan connector:request-direct-profile-2ba --supplier-gln="8714253038889" --trade-item-id="303918432"
 */
class RequestDirectProfile extends Command
{
    protected $signature = 'connector:request-direct-profile-2ba
        {--supplier-gln= : Supplier GLN (default: 8714253038889)}
        {--trade-item-id= : Trade Item ID (default: 303918432)}
        {--lines= : JSON array of lines to request (overrides individual options)}';

    protected $description = 'Request direct profile from 2BA API for specific products';

    private TwoBA $connector;

    public function handle(): int
    {
        $this->connector = new TwoBA();

        try {
            // Parse lines from options
            $lines = $this->parseLines();

            if (empty($lines)) {
                $this->error('No lines provided. Use --supplier-gln and --trade-item-id or --lines option.');
                return Command::FAILURE;
            }

            $this->line('Requesting direct profile from 2BA API...');
            $this->line('Lines to request: ' . json_encode($lines, JSON_PRETTY_PRINT));

            // Make the API request
            $response = $this->connector->selectionProfile()->requestDirectProfile($lines);

            if ($response->failed()) {
                $this->error("API request failed with status: {$response->status()}");
                $this->error("Response: {$response->body()}");
                return Command::FAILURE;
            }

            $responseData = $response->json();

            $this->info('Direct profile request successful!');
            $this->line('Response:');
            $this->line(json_encode($responseData, JSON_PRETTY_PRINT));

            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error('Exception occurred: ' . $e->getMessage());
            $this->line('Stack trace:');
            $this->line($e->getTraceAsString());
            return Command::FAILURE;
        }
    }

    /**
     * Parse lines from command options
     */
    private function parseLines(): array
    {
        // If --lines option is provided, use it directly
        $linesOption = $this->option('lines');
        if ($linesOption) {
            $decoded = json_decode($linesOption, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->error('Invalid JSON in --lines option: ' . json_last_error_msg());
                return [];
            }
            return $decoded;
        }

        // Otherwise, build from individual options
        $supplierGln = $this->option('supplier-gln') ?: '8714253038889';
        $tradeItemId = $this->option('trade-item-id') ?: '303918432';

        return [
            [
                'SupplierGLN' => $supplierGln,
                'TradeItemId' => $tradeItemId,
            ]
        ];
    }
}
