<?php

namespace Daylight\Connector2BA\Service;

use Illuminate\Support\Facades\Log;
use Cerbero\JsonParser\JsonParser;
use function Cerbero\JsonParser\parseJson;
use Cerbero\JsonParser\Decoders\DecodedValue;
use Cerbero\JsonParser\Exceptions\SyntaxException;
use Closure;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Cerbero\LazyJson\LazyJson;
use RuntimeException;
use Illuminate\Support\Arr;

class ETIMReader
{
    private array $memoryLog = [];
    /**
     * Open een bestand als stream, detecteer gzip-compressie.
     *
     * @param string $disk Storage disk name
     * @param string $path Path to the file
     * @return resource Gzip stream or regular file stream
     * @throws RuntimeException if the file cannot be opened
     */
    protected function openResource(string $disk, string $path)
    {
        $full = Storage::disk($disk)->path($path);
        if (!is_file($full) || !is_readable($full)) {
            throw new RuntimeException("Bestand niet leesbaar of bestaat niet: {$full}");
        }

        $fh  = fopen($full, 'rb');
        $hdr = fread($fh, 2) ?: '';
        fclose($fh);

        $isGz = (strlen($hdr) === 2 && ord($hdr[0]) === 0x1F && ord($hdr[1]) === 0x8B);
        if ($isGz) {
            $gz = gzopen($full, 'rb');
            if ($gz === false) {
                throw new RuntimeException("Kon gzip-stream niet openen: {$full}");
            }
            return $gz;
        }

        $fh = fopen($full, 'rb');
        if ($fh === false) {
            throw new RuntimeException("Kon stream niet openen: {$full}");
        }
        return $fh;
    }

    /**
     * Geef een LazyCollection terug op een JSON dot-path (1-pass; verse stream).
     */
    public function stream(string $disk, string $path, string $dotPath): LazyCollection
    {
        $fullPath = Storage::disk($disk)->path($path);
        return LazyJson::from($fullPath, $dotPath);
    }

    /**
     * Stream TradeItems, eventueel gefilterd op leverancier GLN.
     *
     * @param string $disk Storage disk name
     * @param string $path Path to the JSON file
     * @param string|null $supplierIdGln Optional GLN to filter by supplier
     * @return LazyCollection Yields collections with keys: customerRange, gtin, brandName, manufacturerProductNumber, productDescription, etim
     */
    public function streamTradeItems(string $disk, string $path, ?string $supplierIdGln): LazyCollection
    {
        return LazyCollection::make(function () use ($disk, $path, $supplierIdGln) {
            $products = $this->stream(
                $disk,
                $path,
                'Supplier.*.Product.*'
            );
            foreach ($products as $productNode) {
                $arr = $productNode->toArray();
                $dot = Arr::dot($arr);

                $isCustomer = true;

                $etim = collect([]);
                $etimClass = null;
                $attachments = collect([]);
                $description = null;
                $brandInformation = [
                    'series' => null,
                    'type' => null,
                    'name' => $dot['ProductIdentification.BrandName'] ?? $dot['ProductIdentification.ManufacturerName'] ?? null
                ];
                $productName = null;

                if (isset($dot['TradeItem.0.ItemIdentification.SupplierItemNumber'])) {
                    // This is the trade item reference from the seller/owner of the dataset, skip this one.
                    // A reference from the supplier does not contain a TradeItem
                    continue;
                }

                // ETIM data
                if (is_array($dot) && isset($dot['EtimClassification.0.EtimReleaseVersion'])) {
                    $etimClass = $dot['EtimClassification.0.EtimClassCode'] ?? null;
                    $etim = collect($dot)
                        ->mapWithKeys(function ($value, $key) use ($dot) {
                            if (preg_match('/EtimFeatures\.(\d+)\.EtimFeatureCode/', $key, $m)) {
                                $index = $m[1];

                                if (isset($dot["EtimClassification.0.EtimFeatures.$index.EtimValueNumeric"])) {
                                    return [$value => $dot["EtimClassification.0.EtimFeatures.$index.EtimValueNumeric"]];
                                }
                                if (isset($dot["EtimClassification.0.EtimFeatures.$index.ReasonNoValue"])) {
                                    return [$value => $dot["EtimClassification.0.EtimFeatures.$index.ReasonNoValue"]];
                                }
                                if (isset($dot["EtimClassification.0.EtimFeatures.$index.EtimValueRangeLower"])) {
                                    return [$value => $dot["EtimClassification.0.EtimFeatures.$index.EtimValueRangeLower"] . ' - ' . $dot["EtimClassification.0.EtimFeatures.$index.EtimValueRangeUpper"]];
                                }
                                if (isset($dot["EtimClassification.0.EtimFeatures.$index.EtimValueLogical"])) {
                                    return [$value => $dot["EtimClassification.0.EtimFeatures.$index.EtimValueLogical"]];
                                }

                                return [];
                            }
                            return [];
                        });
                }

                // Productdetails
                if (is_array($dot) && isset($dot['ProductDetails.ProductDescriptions.0.DescriptionLanguage'])) {
                    foreach ($dot as $key => $value) {
                        if (preg_match('/ProductDescriptions\.(\d+)\.DescriptionLanguage/', $key, $m)) {
                            $index = $m[1];

                            if ($value === 'nl-NL') {
                                $marketing = "ProductDetails.ProductDescriptions.$index.ProductMarketingText";
                                $unique    = "ProductDetails.ProductDescriptions.$index.UniqueMainProductDescription";
                                $minimal   = "ProductDetails.ProductDescriptions.$index.MinimalProductDescription";

                                if (isset($dot[$marketing])) {
                                    $description = $dot[$marketing]; // Prio 1
                                } elseif (isset($dot[$unique]) && strtolower($dot[$unique]) !== 'geen omschrijving beschikbaar') {
                                    $description = $dot[$unique]; // Prio 2
                                } elseif (isset($dot[$minimal])) {
                                    $description = $dot[$minimal]; // Fallback
                                }

                                if (isset($dot[$minimal]) && is_null($productName)) {
                                    $productName = $dot[$minimal];
                                }

                                break;
                            }

                        }
                    }
                }

                // Branddetails
                if (is_array($dot) && isset($dot['ProductIdentification.BrandName'])) {
                    foreach ($dot as $key => $value) {
                        if (preg_match('/BrandSeries\.(\d+)\.Language/', $key, $m)) {
                            $index = $m[1];

                            if ($value === 'nl-NL') {
                                $brandseries = "ProductIdentification.BrandDetails.0.BrandSeries.$index.BrandSeries";
                                if (isset($dot[$brandseries])) {
                                    $brandInformation['series'] = $dot[$brandseries];
                                }
                            }
                        }
                        if (preg_match('/BrandSeriesVariation\.(\d+)\.Language/', $key, $m)) {
                            $index = $m[1];

                            if ($value === 'nl-NL') {
                                $brandseriesvariation = "ProductIdentification.BrandDetails.0.BrandSeriesVariation.$index.BrandSeriesVariation";
                                if (isset($dot[$brandseriesvariation])) {
                                    $brandInformation['seriesVariant'] = $dot[$brandseriesvariation];
                                }
                            }
                        }
                    }
                }
//
//                if (isset($dot['ProductAttachments.0.AttachmentType'])) {
//                    $preferredLang = 'nl-NL';
//                    $tmp = [];
//                    foreach ($dot as $key => $value) {
//                        // Type
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentType$/', $key, $m)) {
//                            $i = $m[1];
//                            $tmp[$i]['type'] = $value;
//                            continue;
//                        }
//                        // Order
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentOrder$/', $key, $m)) {
//                            $i = $m[1];
//                            $tmp[$i]['order'] = (int) $value;
//                            continue;
//                        }
//                        // URI
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentDetails\.(\d+)\.AttachmentUri$/', $key, $m)) {
//                            [$i, $j] = [$m[1], $m[2]];
//                            $tmp[$i]['details'][$j]['uri'] = $value;
//                            continue;
//                        }
//                        // Languages (array)
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentDetails\.(\d+)\.AttachmentLanguage\.\d+$/', $key, $m)) {
//                            [$i, $j] = [$m[1], $m[2]];
//                            $tmp[$i]['details'][$j]['lang'][] = $value;
//                            continue;
//                        }
//                        // TypeSpecification: Language
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentDetails\.(\d+)\.AttachmentTypeSpecification\.(\d+)\.Language$/', $key, $m)) {
//                            [$i, $j, $k] = [$m[1], $m[2], $m[3]];
//                            $tmp[$i]['details'][$j]['spec'][$k]['lang'] = $value;
//                            continue;
//                        }
//                        // TypeSpecification: Label
//                        if (preg_match('/^ProductAttachments\.(\d+)\.AttachmentDetails\.(\d+)\.AttachmentTypeSpecification\.(\d+)\.AttachmentTypeSpecification$/', $key, $m)) {
//                            [$i, $j, $k] = [$m[1], $m[2], $m[3]];
//                            $tmp[$i]['details'][$j]['spec'][$k]['label'] = $value;
//                            continue;
//                        }
//                    }
//
//                    $attachments = collect($tmp)
//                        ->filter(fn ($att) => !empty($att['type']))
//                        ->sortBy('order')
//                        ->flatMap(function ($att) use ($preferredLang) {
//                            $type    = $att['type'];
//                            $details = collect($att['details'] ?? []);
//
//                            $hasPreferred = $details->contains(fn ($d) => in_array($preferredLang, $d['lang'] ?? [], true));
//
//                            $filtered = $details->filter(function ($d) use ($preferredLang, $hasPreferred) {
//                                return $hasPreferred ? in_array($preferredLang, $d['lang'] ?? [], true) : true;
//                            });
//
//                            return $filtered->map(function ($d) use ($type, $preferredLang) {
//                                $label = collect($d['spec'] ?? [])
//                                    ->firstWhere('lang', $preferredLang)['label'] ?? null;
//
//                                $lang = in_array($preferredLang, $d['lang'] ?? [], true)
//                                    ? $preferredLang
//                                    : ($d['lang'][0] ?? null);
//
//                                return [
//                                    'type'  => $type,
//                                    'uri'   => $d['uri'] ?? null,
//                                    'label' => $label,
//                                    'lang'  => $lang,
//                                ];
//                            });
//                        })
//                        ->filter(fn ($x) => !empty($x['uri']))
//                        ->groupBy('type')
//                        ->map(fn ($items) => $items->unique('uri')->values());
//                }

                $base = collect([
                    'isCustomer' => $isCustomer,
                    'internalCode' => $dot['TradeItem.0.ItemIdentification.SupplierItemNumber'] ?? null,
                    'discountGroup' => $dot['TradeItem.0.ItemIdentification.DiscountGroupId'] ?? null,
                    'gln' => $supplierIdGln ?? ($dot['ProductIdentification.ManufacturerIdGln'] ?? null),
                    'gtin' => $dot['ProductIdentification.ProductGtin.0'] ?? null,
                    'brandName' => $dot['ProductIdentification.ManufacturerName'] ?? null,
                    'brandInformation' => $brandInformation,
                    'manufacturerProductNumber' => $dot['ProductIdentification.ManufacturerProductNumber'] ?? null,
                    'productName' => $productName,
                    'productDescription' => $description,
                    'etimClass' => $etimClass,
                    'etim' => $etim,
                    'attachments' => $attachments,
                ]);

                yield $base;
            }
        });
    }

    // Check to handle the import for customer data (HEG) or supplier data (2BA)
    private function isCustomer(bool $etim, ?string $supplierIdGln): bool
    {
        return is_null($supplierIdGln)
            ? $etim
            : !($etim && (string)$supplierIdGln === (string) config('services.connector.2ba.supplier_id_gln'));
    }

    private function nodeToArray($node): array
    {
        if ($node instanceof LazyCollection) {
            $node = $node->collect();
        }
        if ($node instanceof Collection) {
            return $node->map(fn ($v) => $this->nodeToArray($v))->all();
        }
        if (is_array($node)) {
            return array_map(fn ($v) => $this->nodeToArray($v), $node);
        }
        return $node;
    }

    /**
     * Probeer een waarde als JSON te parsen, of geef null terug bij fouten.
     */
    protected function col($value): Collection
    {
        if ($value instanceof LazyCollection) return $value->collect();
        if ($value instanceof Collection)     return $value;
        return collect($value);
    }

    /**
     * Verwerk TradeItems in chunks, met een handler functie.
     *
     * @param string $disk Storage disk name
     * @param string $path Path to the JSON file
     * @param string|null $supplierIdGln Optional GLN to filter by supplier
     * @param int $chunkSize Number of items per chunk
     * @param Closure $handler Function to handle each chunk of items
     */
    public function processTradeItems(string $disk, string $path, ?string $supplierIdGln, int $chunkSize, Closure $handler): void
    {
        $this->logMemory('Process start');

        $stream = $this->streamTradeItems($disk, $path, $supplierIdGln)
            ->chunk($chunkSize);

        $chunkNo = 0;

        foreach ($stream as $chunk) {
            /** @var LazyCollection $chunk */
            $batch = $chunk->collect()
                ->map(fn($row) => $row instanceof Collection ? $row->all() : (array) $row)
                ->all();

            $chunkNo++;

            $this->logMemory("After chunk $chunkNo collect", [
                'chunk_size' => count($batch),
            ]);

            try {
                $handler($batch);
                $this->logMemory("After chunk $chunkNo handler");
            } catch (\Throwable $e) {
                Log::error('ETIM handler failed', [
                    'chunk'   => $chunkNo,
                    'message' => $e->getMessage(),
                    'memory_usage' => $this->getCurrentMemoryUsage(),
                ]);
                throw $e;
            }

            if ($chunkNo % 10 === 0) {
                gc_collect_cycles();
                $this->logMemory("After GC (chunk $chunkNo)");
            }
        }
    }

    /**
     * Log messages - uses Laravel Log if available, otherwise falls back to error_log
     */
    protected function log(string $level, string $message, array $context = []): void
    {
        if (class_exists('Illuminate\Support\Facades\Log')) {
            \Illuminate\Support\Facades\Log::$level($message, $context);
        } else {
            // Fallback for standalone usage
            $contextStr = !empty($context) ? ' ' . json_encode($context) : '';
            error_log("[$level] $message$contextStr");
        }
    }

    private function logMemory(string $stage, array $context = []): void
    {
        $current = memory_get_usage(true);
        $peak = memory_get_peak_usage(true);

        $memoryData = [
            'stage' => $stage,
            'current_mb' => round($current / 1024 / 1024, 2),
            'peak_mb' => round($peak / 1024 / 1024, 2),
            'timestamp' => microtime(true),
        ];

        $this->memoryLog[] = array_merge($memoryData, $context);

        Log::info('Memory usage', $memoryData + $context);
    }

    private function getCurrentMemoryUsage(): array
    {
        return [
            'current_mb' => round(memory_get_usage(true) / 1024 / 1024, 2),
            'peak_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2),
        ];
    }

    private function logMemorySummary(): void
    {
        if (empty($this->memoryLog)) return;

        $first = $this->memoryLog[0];
        $last = end($this->memoryLog);
        $maxPeak = max(array_column($this->memoryLog, 'peak_mb'));

        Log::info('Memory usage summary', [
            'start_memory_mb' => $first['current_mb'],
            'end_memory_mb' => $last['current_mb'],
            'memory_growth_mb' => round($last['current_mb'] - $first['current_mb'], 2),
            'max_peak_mb' => $maxPeak,
            'duration_seconds' => round($last['timestamp'] - $first['timestamp'], 2),
        ]);
    }
}
