<?php

namespace Daylight\Connector2BA\Jobs;

use App\Jobs\SyncMeilisearchSettingsJob;
use Daylight\Core\Models\AttributeValue;
use Daylight\Core\Models\Category;
use Daylight\Core\Models\Variant;
use Daylight\Core\Models\Attribute;
use Daylight\Core\Models\Brand;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;use function Symfony\Component\String\u;

class UpdateVariant implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private array $variantData = [];

    public function __construct(
        private Collection $variant
    )
    {
        $this->variantData = $this->variant?->toArray();
//        dd($this->variantData);
    }

    public function handle(): ?string
    {
        $variant = Variant::where('ean', $this->variantData['gtin'])->first();

        if (!$variant) {
            // remove leading 0 and try again
            $variant = Variant::where('ean', ltrim($this->variantData['gtin'], '0'))->first();

            if (!$variant) {
                return false;
            }
        }

        $product = $variant->product;

        if ($this->variantData['description']) {
            $translation = $product->translations()?->where([
                ['product_id', '=', $product->id],
                ['locale', '=', 'nl']
            ])?->first();

            if ($translation !== null) {
                $translation->description = $this->variantData['description'];
//                $translation->name = $this->product->title;
                $translation->save();
            } else {
                $product->translations()->create(
                    [
                        'locale' => 'nl',
//                        'slug' => Str::slug($this->product->title . '-' . $this->product->sku),
                        'name' => substr($this->variantData['description'], 0, 220),
                        'description' => $this->variantData['description'],
                    ]
                );
            }
        }

//        $variant->optionValues()->sync([]);

        // Add data to variant
        $variant->update([
            'gln_manufacturer' => $this->variantData['gln'] ?? null,
            'gtin_product' => $this->variantData['gtin'] ?? null,
            'productcode_manufacturer' => $this->variantData['product_code'] ?? null,
            'type_manufacturer' => null,
            'series' => null,
            'unspsc_code' => null,
            '2ba_synced_at' => Carbon::now(),
        ]);
        $variant->save();

        // Attributes
        $attributes_to_sync = self::handleAttributes($variant, $this->variantData);
        $product->attributeValues()->syncWithoutDetaching($attributes_to_sync);

        // Categories
        $categories_to_sync = self::handleCategories($variant, ($this->variantData['group_name_nl'] ?? null));
        if (!is_null($categories_to_sync) && count($categories_to_sync) > 0) {
            $product->categories()->syncWithoutDetaching($categories_to_sync);
        }

        // Brand
        $product->update(
            [
                'brand_id' => self::getBrandId($this->variantData['brand_name'] ?? null)
            ]
        );
        $product->save();

//        // Dispatch Meilisearch settings sync job
//        SyncMeilisearchSettingsJob::dispatch();

        return $variant->id;
    }

    /**
     * Create or update variant attributes
     *
     * @param Variant $variant
     * @param array $variantData
     *
     * @return void
     */
    private static function handleAttributes(Variant $variant, array $variantData): array
    {
        $return = [];

        if ($variantData['etim_features'] && $variantData['group_name_nl']) {

            $loop = 0;
            $existingValue = null;
            foreach($variantData['etim_features'] as $etimFeature) {
                if (!$etimFeature) {
                    continue;
                }
                if ($etimFeature['processed_value'] === false) {
                    continue;
                }

                $attribute = Attribute::firstOrCreate(
                    ['key' => Str::slug($etimFeature['feature_code'])],
                    [
                        'filterable' => 1,
                        'visible' => 1
                    ]
                );

                $attribute->translations()->updateOrCreate(
                    [
                        'attribute_id' => $attribute->id,
                        'locale' => 'nl',
                    ],
                    [
                        'name' => $etimFeature['feature_name_nl'],
                    ]
                );

                $existingValue = self::findExistingAttributeValueByLabel($attribute, trim($etimFeature['processed_value'] . ' ' . $etimFeature['unit_symbol']), 'nl');
                if (is_null($existingValue)) {
                    $attributeValue = $attribute->values()->create(
                        [
                            'attribute_id' => $attribute->id,
                        ],
                        []
                    );
                    $existingValue = $attributeValue;
                } else {
                    $attributeValue = $existingValue;
                }

                $attributeValue->translations()->updateOrCreate(
                    [
                        'attribute_value_id' => $attributeValue->id,
                        'locale' => 'nl',
                        'name' => trim($etimFeature['processed_value'] . ' ' . $etimFeature['unit_symbol']),
                    ],
                    []
                );

                // Attach the attribute value to the product if not already attached
                $return[] = $attributeValue->id;
            }
        }

        return $return;
    }

    /**
     * Find an existing option with the same label
     * Searches globally for options with matching name to prevent duplicates
     *
     * @param Daylight\Core\Models\Attribute $attribute
     * @param ?string $label
     * @param string $locale
     *
     * @return AttributeValue|null
     */
    private static function findExistingAttributeValueByLabel(Attribute $attribute, ?string $label, ?string $locale = 'en'): ?AttributeValue
    {
        // Search globally for an attributevalue by attribute with this label name
        // This prevents creating duplicate attributes with the same name
        $attributeValue = AttributeValue::whereHas('translations', function ($query) use ($attribute, $label, $locale) {
            $query->where('name', $label);
            $query->where('locale', $locale);
            $query->where('attribute_id', $attribute->id);
        })->first();

        return $attributeValue;
    }

    private static function handleCategories(Variant $variant, ?string $etimGroup): ?array
    {
        if (
            class_exists(\App\Models\Category::class, 'buildCustomCategory')
            && $variant
        ) {
            $arr = \App\Models\Category::buildCustomCategory($variant, $etimGroup);
            if (is_null($arr)) {
                return null;
            }
            return array_filter($arr, fn($value) => !is_null($value));
        }

        return null;
    }

    private static function getBrandId(?string $name): ?int
    {
        if (is_null($name)) {
            return null;
        }

        return Brand::firstOrCreate(
            [
                'slug' => Str::slug($name),
                'name' => $name
            ]
        )?->id;
    }
}
