<?php

namespace Daylight\Connector2BA\Http\Auth;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Saloon\Contracts\Authenticator;
use Saloon\Http\PendingRequest;

class OAuth2Authenticator implements Authenticator
{
    private string $accessTokenCacheKey;
    private string $refreshTokenCacheKey;

    public function __construct(
        private string $clientId,
        private string $clientSecret,
        private ?string $username = null,
        private ?string $password = null
    ) {
        $this->accessTokenCacheKey = 'connector_2ba_access_token';
        $this->refreshTokenCacheKey = 'connector_2ba_refresh_token';
    }

    public function set(PendingRequest $pendingRequest): void
    {
        $accessToken = $this->getAccessToken();

        if ($accessToken) {
            $pendingRequest->headers()->add('Authorization', 'Bearer ' . $accessToken);
        }
    }

    private function getAccessToken(): ?string
    {
        // Try to get cached access token
        $cachedToken = Cache::get($this->accessTokenCacheKey);

        if ($cachedToken && $this->isTokenValid($cachedToken)) {
            return $cachedToken['access_token'];
        }

        // Try to refresh with refresh token
        $refreshToken = Cache::get($this->refreshTokenCacheKey);
        if ($refreshToken) {
            $newToken = $this->refreshAccessToken($refreshToken);
            if ($newToken) {
                return $newToken['access_token'];
            }
        }

        // Get new token using password grant (if credentials available)
        if ($this->username && $this->password) {
            $newToken = $this->getPasswordGrantToken();
            if ($newToken) {
                return $newToken['access_token'];
            }
        }

        logger()->error('Failed to get 2BA access token: No valid authentication method available');
        return null;
    }

    private function getPasswordGrantToken(): ?array
    {
        try {
            $response = Http::asForm()->post('https://authorize.2ba.nl/OAuth/Token', [
                'grant_type' => 'password',
                'username' => $this->username,
                'password' => $this->password,
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
            ]);

            if ($response->failed()) {
                logger()->error('2BA OAuth password grant failed', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return null;
            }

            $tokenData = $response->json();
            $this->cacheTokens($tokenData);

            logger()->info('2BA OAuth password grant successful');
            return $tokenData;

        } catch (\Exception $e) {
            logger()->error('2BA OAuth password grant exception: ' . $e->getMessage());
            return null;
        }
    }

    private function refreshAccessToken(string $refreshToken): ?array
    {
        try {
            $response = Http::asForm()->post('https://authorize.2ba.nl/OAuth/Token', [
                'grant_type' => 'refresh_token',
                'refresh_token' => $refreshToken,
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
            ]);

            if ($response->failed()) {
                logger()->warning('2BA OAuth token refresh failed', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);

                // Remove invalid refresh token
                Cache::forget($this->refreshTokenCacheKey);
                return null;
            }

            $tokenData = $response->json();
            $this->cacheTokens($tokenData);

            logger()->info('2BA OAuth token refresh successful');
            return $tokenData;

        } catch (\Exception $e) {
            logger()->error('2BA OAuth token refresh exception: ' . $e->getMessage());
            Cache::forget($this->refreshTokenCacheKey);
            return null;
        }
    }

    private function cacheTokens(array $tokenData): void
    {
        // Cache access token with expiry (subtract 60 seconds for safety margin)
        $expiresIn = ($tokenData['expires_in'] ?? 3600) - 60;
        $accessTokenData = [
            'access_token' => $tokenData['access_token'],
            'expires_at' => time() + $expiresIn,
        ];

        Cache::put($this->accessTokenCacheKey, $accessTokenData, $expiresIn);

        // Cache refresh token (typically valid much longer)
        if (isset($tokenData['refresh_token'])) {
            Cache::put($this->refreshTokenCacheKey, $tokenData['refresh_token'], 86400 * 30); // 30 days
        }
    }

    private function isTokenValid(array $tokenData): bool
    {
        return isset($tokenData['expires_at']) && $tokenData['expires_at'] > time();
    }

    public function clearTokens(): void
    {
        Cache::forget($this->accessTokenCacheKey);
        Cache::forget($this->refreshTokenCacheKey);
    }

    public function setCredentials(string $username, string $password): void
    {
        $this->username = $username;
        $this->password = $password;
    }

    /**
     * Force refresh the access token
     */
    public function refreshToken(): ?string
    {
        $refreshToken = Cache::get($this->refreshTokenCacheKey);

        if ($refreshToken) {
            $newToken = $this->refreshAccessToken($refreshToken);
            return $newToken['access_token'] ?? null;
        }

        // Fall back to password grant if no refresh token
        if ($this->username && $this->password) {
            $newToken = $this->getPasswordGrantToken();
            return $newToken['access_token'] ?? null;
        }

        return null;
    }
}
