<?php

namespace Daylight\ConnectorCash\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\ConnectorCash\Http\Cash;
use Daylight\ConnectorCash\Jobs\UpdateAddress;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Connector\Support\ConnectorLogger;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\ConnectorCash\Jobs\UpdateProduct;
use Daylight\ConnectorCash\Jobs\UpdateCustomer;
use Illuminate\Support\Facades\Log;
use Illuminate\Bus\PendingBatch;
use Daylight\Core\Models\Product;

class UpdateAddresses extends Command
{
    protected $signature = 'connector:update-addresses {--connector=} {--force}';

    protected $description = 'Get addresses from the ERP';

    protected BaseConnector|Cash $connector;

    public function handle(ConnectorRegistry $registry): int
    {
        $iteration = 1;

        $this->connector = $this->getConnector(
            $registry->all(),
            $this->option('connector') ?? ''
        );

        if (!$this->connector) {
            $this->error('No active connector found.');
            return Command::FAILURE;
        }

        if (!method_exists($this->connector, 'getAddressMapper')) {
            $this->error('No address mapper found in ' .  $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start import for ' . $this->connector->getName());

        LazyCollection::make(function () use (&$iteration) {
            //do {
            try {
                $results = $this->getItems();
            } catch (\Exception $e) {
                $this->error('Exception ' .  $e->getMessage());
                $results = collect();
            }

            if (count($results) === 0) {
                return;
            }

            yield $results;

            $iteration++;
            //} while ($results->isNotEmpty());
        })->each(function (Collection $addresses, $index) {
            $jobs = [];

            foreach ($addresses as $address) {
                $jobs[] = new UpdateAddress($address);
            }

            Bus::batch($jobs)
                ->onQueue($this->connector->getKey())
                ->dispatch();

            unset($jobs);
            unset($addresses);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        return Command::SUCCESS;
    }

    protected function getItems(int $page = 1, int $perPage = 250): ?Collection
    {
        return $this->connector
            ->addresses()
            ->all($this->connector->getAddressMapper(), [])
            ->dtoOrFail();
    }

    private function getConnector(Collection $connectors, string $connectorKey): null|BaseConnector|Cash
    {
        if ($connectors[$connectorKey]) {
            $class = $connectors[$connectorKey]->connector;

            return new $class;
        }

        return null;
    }
}
