<?php

namespace Daylight\ConnectorCash\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\ConnectorCash\Http\Cash;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Connector\Support\ConnectorLogger;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\ConnectorCash\Jobs\UpdateProduct;
use Daylight\ConnectorCash\Jobs\UpdateCustomer;
use Illuminate\Support\Facades\Log;
use Illuminate\Bus\PendingBatch;
use Daylight\Core\Models\Product;

class UpdateCustomers extends Command
{
    protected $signature = 'connector:update-customers {--connector=} {--force}';

    protected $description = 'Get Customers from the ERP';

    protected BaseConnector|Cash $connector;

    public function handle(ConnectorRegistry $registry): int
    {
        $iteration = 1;

        $this->connector = $this->getConnector(
            $registry->all(),
            $this->option('connector') ?? ''
        );

        if (!$this->connector) {
            $this->error('No active connector found.');
            return Command::FAILURE;
        }

        if (!method_exists($this->connector, 'getCustomerMapper')) {
            $this->error('No customer mapper found in ' .  $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start import for ' . $this->connector->getName());

        LazyCollection::make(function () use (&$iteration) {
            //do {
            try {
                $results = $this->getItems();
            } catch (\Exception $e) {
                $this->error('Exception ' .  $e->getMessage());
                $results = collect();
            }

            if (count($results) === 0) {
                return;
            }

            yield $results;

            $iteration++;
            //} while ($results->isNotEmpty());
        })->each(function (Collection $products, $index) {
            $jobs = [];

            foreach ($products as $product) {
                $jobs[] = new UpdateCustomer($product);
            }

            Bus::batch($jobs)
                ->onQueue($this->connector->getKey())
                ->dispatch();

            unset($jobs);
            unset($products);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        return Command::SUCCESS;
    }

    protected function getItems(int $page = 1, int $perPage = 250): ?Collection
    {
        return $this->connector
            ->customers()
            ->all($this->connector->getCustomerMapper(), ['pageNumber' => $page, 'perPage' => $perPage])
            ->dtoOrFail();
    }

    private function getConnector(Collection $connectors, string $connectorKey): null|BaseConnector|Cash
    {
        if ($connectors[$connectorKey]) {
            $class = $connectors[$connectorKey]->connector;

            return new $class;
        }

        return null;
    }
}
