<?php

namespace Daylight\ConnectorCash;

use GuzzleHttp\Psr7\Uri;
use Psr\Http\Message\UriInterface;

class CustomUriGenerator implements UriInterface
{
    private string $scheme;
    private string $userInfo;
    private string $host;
    private ?int $port;
    private string $path;
    private string $query;
    private string $fragment;
    private ?string $composedComponents = null;

    public function __construct(string $uri)
    {
        // Parse the URI using Guzzle's Uri class to get all components
        $parsedUri = new Uri($uri);
        
        $this->scheme = $parsedUri->getScheme();
        $this->userInfo = $parsedUri->getUserInfo();
        $this->host = $parsedUri->getHost();
        $this->port = $parsedUri->getPort();
        $this->query = $parsedUri->getQuery();
        $this->fragment = $parsedUri->getFragment();
        
        // The key difference: preserve literal pipe characters in the path
        $this->path = str_replace('%7C', '|', $parsedUri->getPath());
    }

    public function __toString(): string
    {
        if ($this->composedComponents === null) {
            $this->composedComponents = Uri::composeComponents(
                $this->scheme,
                $this->getAuthority(),
                $this->path,
                $this->query,
                $this->fragment
            );
        }
        
        return $this->composedComponents;
    }

    public function getScheme(): string
    {
        return $this->scheme;
    }

    public function getAuthority(): string
    {
        $authority = $this->host;
        if ($this->userInfo !== '') {
            $authority = $this->userInfo . '@' . $authority;
        }
        
        if ($this->port !== null) {
            $authority .= ':' . $this->port;
        }
        
        return $authority;
    }

    public function getUserInfo(): string
    {
        return $this->userInfo;
    }

    public function getHost(): string
    {
        return $this->host;
    }

    public function getPort(): ?int
    {
        return $this->port;
    }

    public function getPath(): string
    {
        return $this->path;
    }

    public function getQuery(): string
    {
        return $this->query;
    }

    public function getFragment(): string
    {
        return $this->fragment;
    }

    public function withScheme($scheme): UriInterface
    {
        if ($this->scheme === $scheme) {
            return $this;
        }
        
        $new = clone $this;
        $new->scheme = $scheme;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withUserInfo($user, $password = null): UriInterface
    {
        $info = $user;
        if ($password !== null && $password !== '') {
            $info .= ':' . $password;
        }
        
        if ($this->userInfo === $info) {
            return $this;
        }
        
        $new = clone $this;
        $new->userInfo = $info;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withHost($host): UriInterface
    {
        if ($this->host === $host) {
            return $this;
        }
        
        $new = clone $this;
        $new->host = $host;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withPort($port): UriInterface
    {
        if ($this->port === $port) {
            return $this;
        }
        
        $new = clone $this;
        $new->port = $port;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withPath($path): UriInterface
    {
        // Preserve pipe characters in the path
        $pathWithPipes = str_replace('%7C', '|', $path);
        
        if ($this->path === $pathWithPipes) {
            return $this;
        }
        
        $new = clone $this;
        $new->path = $pathWithPipes;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withQuery($query): UriInterface
    {
        if ($this->query === $query) {
            return $this;
        }
        
        $new = clone $this;
        $new->query = $query;
        $new->composedComponents = null;
        
        return $new;
    }

    public function withFragment($fragment): UriInterface
    {
        if ($this->fragment === $fragment) {
            return $this;
        }
        
        $new = clone $this;
        $new->fragment = $fragment;
        $new->composedComponents = null;
        
        return $new;
    }
}
