<?php

namespace Daylight\ConnectorCash\Jobs;

use App\Models\User;
use Daylight\Core\Models\Customer;
use Daylight\ConnectorCash\Entities\Customer as CustomerEntity;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;
use Daylight\Core\Models\Enums\CustomerType;
use Daylight\Core\Models\Enums\AddressType;
use Illuminate\Support\Facades\Log;

class UpdateCustomer implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public CustomerEntity $customer,
    ) {
       //
    }

    public function handle(): ?string
    {
        if (!$this->customer->external_id || !$this->customer->email || $this->customer->email == '-') {
            Log::error('Cannot create customer or user: ', [$this->customer->external_id, $this->customer->email]);
            return false;
        }

        $customer = Customer::updateOrCreate([
            'external_id' => $this->customer->external_id,
        ], [
            'type' => CustomerType::COMPANY->value, // Fixed to companies for now
            'name' => $this->customer->name,
            'active' => ($this->customer->is_webshop == 'J') ? 1 : 0,
        ]);


        if ($this->customer->address) {
            $customer->addresses()->firstOrCreate([
                'type' => AddressType::BILLING->value,
            ], [
                'default' => 1,
                'name' => $this->customer->name,
                'address_line_1' => $this->customer->address,
                'postal_code' => $this->customer->splitBillingZipCity()[0],
                'city' => $this->customer->splitBillingZipCity()[1],
                'country_id' => $this->customer->getCountryCode()->id ?? 26,
                'invoicing_email' => $this->customer->invoiceEmail,
            ]);
        }

        return $customer->id;
    }
}
