<?php

namespace Daylight\ConnectorCash\Jobs;

use Daylight\Core\Models\Order;
use Daylight\ConnectorCash\Entities\Order as OrderEntity;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Daylight\Core\Models\Customer;
use Daylight\Core\Models\TaxClass;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class UpdateOrder implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public OrderEntity $order,
    ) {
        //
    }

    public function handle(): ?string
    {
        if (!$this->order->external_id || !$this->order->customer_external_id || $this->order->items->isEmpty()) {
            Log::error('Cannot create order: ', [$this->order->external_id , $this->order->customer_external_id]);
            return false;
        }

        $customer = Customer::query()->where('external_id', $this->order->customer_external_id)->first();
        if (!$customer) {
            Log::error('Cannot find customer: ', [$this->order->customer_external_id]);
            return false;
        }

        $user_id = $customer->user?->firstWhere('number', $this->usernumber)?->id;

        $subtotal_value = 0;
        $shipping_value = 0;
        $tax_value = 0;
        $total_value = 0;

        $orderItems = $this->order->items->map(
            function ($item) use ($customer, &$subtotal_value, &$shipping_value, &$tax_value) {
                $variant = Variant::query()
                    ->where('sku', $item->sku)
                    ->with('taxClass.taxRates')
                    ->first();
                $item->variant_id = $variant?->id;
                $item->name = $variant->product->name ?? '';

                $item->sale_price_final = ($item->discount && $item->discount > 0) ? $item->sale_price * (1 - ($item->discount / 100)) : $item->sale_price;

                if ($variant) {
                    $country_id = $customer->defaultBillingAddress()?->country_id ?? 26;

                    $taxClass = $variant->taxClass->taxRates->firstWhere('country_id', $country_id);

                    $tax = ($item->quantity * ($item->sale_price_final * ((float)$taxClass->rate / 100)));
                    $item->tax = $tax;
                    $item->tax_breakdown = [
                        [
                            'name' => $taxClass->name,
                            'rate' => $taxClass->rate,
                            'amount' => $tax,
                        ],
                    ];
                } else {
                    $tax = ($item->quantity * ($item->sale_price_final * (21 / 100)));
                    $item->tax = $tax;
                    $item->tax_breakdown = [
                        [
                            'name' => 'BTW',
                            'rate' => '21.0000',
                            'amount' => $tax,
                        ],
                    ];
                }
                $subtotal_value += ($item->quantity * $item->sale_price_final);
                $tax_value += $tax;

                return $item;
            }
        );

        // Totals
        $total_value = ($shipping_value + $tax_value + $subtotal_value);

        $order = Order::updateOrCreate(
            [
                'external_id' => (int)$this->order->external_id
            ],
            [
                'customer_id' => $customer->id,
                'user_id' => $user_id,
                'number' => $this->order->external_id,
                'status' => $this->order->status,
                'subtotal' => $subtotal_value,
                'shipping' => $shipping_value,
                'tax' => $tax_value,
                'total' => $total_value,
                'reference' => $this->order->reference,
                'delivery_date' => $this->order->delivery_date,
                'payment_method' => $this->order->payment_condition ?? 'invoice',
                'created_at' => $this->order->order_date
            ]
        );

        foreach ($orderItems as $item) {
            $order->items()->updateOrCreate(
                [
                    'sku' => $item->sku
                ],
                [
                    'variant_id' => $item->variant_id,
                    'status' => $item->status,
                    'sku' => $item->sku,
                    'name' => $item->name,
                    'discount' => $item->discount,
                    'price' => $item->sale_price,
                    'price_final' => $item->sale_price_final,
                    'quantity' => $item->quantity,
                    'tax' => $item->tax,
                    'tax_breakdown' => $item->tax_breakdown,
                    'delivery_date' => $item->delivery_date,
                    'seq' => $item->seq,
                    'created_at' => $this->order->order_date
                ]
            );
        }

        return $this->order->external_id;
    }

    public function uniqueId(): string
    {
        return $this->order->external_id;
    }
}
