<?php

namespace Daylight\ConnectorCash\Mappers;

use Daylight\Connector\Contracts\MapperInterface;
use Daylight\Core\Models\Customer;
use Daylight\ConnectorCash\Entities\OrderItem as OrderItemEntity;
use Illuminate\Support\Facades\Date;
use Carbon\Carbon;
use Daylight\ConnectorCash\Enums\OrderStatus;
use NumberFormatter;

class OrderItemMapper implements MapperInterface
{
    /**
     * Map the data to the format required by the connector.
     *
     * @param array $data
     * @return array
     */
    public function map(array $data): OrderItemEntity
    {
        return new OrderItemEntity(
            $data['F2400'] ?? null,
            isset($data['F2002']) ? ((float)$data['F2002']) : 0,
            $data['F2260'] ?? null,
            isset($data['F2267']) ? $this->formatPrice($data['F2267']) : 0,
            isset($data['F2421']) ? $this->formatPrice($data['F2421']) : 0,
            $data['F2268'] ?? null,
            isset($data['F2402']) ? Carbon::createFromFormat('d-m-Y', $data['F2402']) : null,
            isset($data['F2410']) ? ((float)$data['F2410'] / 100) : 1,
            (int)$data['F2426'] ?? 0,
            (int)$data['F3325'] ?? 0,
            OrderStatus::fromCode($data['F2404'] ?? 'L')?->label(), //L = lopende order, V = verzamelbon, G = gereedgemelde verkooporder, B Verkooporderbevestiging afgedrukt, M Magazijnbon afgedrukt, P Pakbon afgedrukt, F Factureerbaar gestelde verkooporder
        );
    }

    public function mapWithAttributes(array $data, array $attributes = []): OrderItemEntity
    {
        $product = $this->map($data);
        foreach ($attributes as $key => $value) {
            $product->{$key} = $value;
        }
        return $product;
    }

    private function formatPrice(string $price = "0"): int
    {
        $formatter = new NumberFormatter('nl_NL', NumberFormatter::DECIMAL);

        return $formatter->parse($price) * 100 ?? 0;
    }
}
