<?php

namespace Daylight\ConnectorCash\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\ConnectorCash\Http\Cash;
use Daylight\ConnectorCash\Jobs\UpdatePriceAgreement;
use Daylight\ConnectorCash\Jobs\UpdatePriceList;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Connector\Support\ConnectorLogger;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\ConnectorCash\Jobs\UpdateProduct;
use Daylight\ConnectorCash\Jobs\UpdateCustomer;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Bus\PendingBatch;
use Daylight\Core\Models\Product;
use Daylight\ConnectorCash\Mappers\PriceListMapper;
use Carbon\Carbon;

class UpdatePriceAgreements extends Command
{
    protected $signature = 'connector:update-priceagreements {--connector=} {--force}';

    protected $description = 'Get Priceagreements from the ERP';

    protected BaseConnector|Cash $connector;

    public function handle(ConnectorRegistry $registry): int
    {
        $iteration = 1;
        $syncTimestamp = now();
        $processedAgreements = collect();

        $this->connector = $this->getConnector(
            $registry->all(),
            $this->option('connector') ?? ''
        );

        if (!$this->connector) {
            $this->error('No active connector found.');
            return Command::FAILURE;
        }

        if (!method_exists($this->connector, 'getPriceAgreementMapper')) {
            $this->error('No customer mapper found in ' .  $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start import for ' . $this->connector->getName());

        LazyCollection::make(function () use (&$iteration) {
            //do {
            try {
                $results = $this->getItems();
            } catch (\Exception $e) {
                $this->error('Exception ' .  $e->getMessage());
                $results = collect();
            }

            if (count($results) === 0) {
                return;
            }

            yield $results;

            $iteration++;
            //} while ($results->isNotEmpty());
        })->each(function (Collection $products, $index) use (&$processedAgreements) {
            $jobs = [];

            foreach ($products as $product) {
                $processedAgreements->push([
                    'sku' => $product->sku,
                    'customer_external_id' => $product->customer_external_id,
                    'date' => $product->date,
                ]);

                $jobs[] = new UpdatePriceAgreement($product);
            }

            Bus::batch($jobs)
                ->name('cash:priceagreements:iteration:' . ($index + 1))
                ->onQueue($this->connector->getKey())
                ->dispatch();

            unset($jobs);
            unset($products);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        $this->cleanupOldAgreements($processedAgreements, $syncTimestamp);

        return Command::SUCCESS;
    }

    protected function getItems(int $page = 1, int $perPage = 250): ?Collection
    {
        return $this->connector
            ->prices()
            ->all($this->connector->getPriceAgreementMapper(), ['pageNumber' => $page, 'perPage' => $perPage, 'endpoint' => 2274])
            ->dtoOrFail();
    }

    /**
     * Verwijder price agreements die niet meer in het ERP zitten
     */
    protected function cleanupOldAgreements(Collection $processedAgreements, Carbon $syncTimestamp): void
    {
        if ($processedAgreements->isEmpty()) {
            $this->line('No agreements processed, skipping cleanup.');
            return;
        }

        $this->line('Starting cleanup of old price agreements...');

        $customerExternalIds = $processedAgreements->pluck('customer_external_id')->unique()->filter();

        $customerIds = DB::table('customers')
            ->whereIn('external_id', $customerExternalIds)
            ->pluck('id')
            ->toArray();

        if (empty($customerIds)) {
            return;
        }

        $skus = $processedAgreements->pluck('sku')->unique()->filter();

        $variantIds = DB::table('variants')
            ->whereIn('sku', $skus)
            ->pluck('id')
            ->toArray();

        if (empty($variantIds)) {
            return;
        }

        $validKeys = $processedAgreements->map(function ($agreement) {
            $customerId = DB::table('customers')
                ->where('external_id', $agreement['customer_external_id'])
                ->value('id');

            $variantId = DB::table('variants')
                ->where('sku', $agreement['sku'])
                ->value('id');

            if (!$customerId || !$variantId) {
                return null;
            }

            $startsAt = $agreement['date']
                ? Carbon::parse($agreement['date'])->startOfDay()->toDateTimeString()
                : 'null';

            return "{$customerId}:{$variantId}:{$startsAt}";
        })->filter()->unique();

        $existingPrices = daylightModel('price')::query()
            ->whereIn('customer_id', $customerIds)
            ->whereIn('variant_id', $variantIds)
            ->whereNull('customer_group_id')
            ->whereNotNull('customer_id')
            ->get();

        $deletedCount = 0;

        foreach ($existingPrices as $price) {
            $startsAt = $price->starts_at
                ? $price->starts_at->startOfDay()->toDateTimeString()
                : 'null';

            $key = "{$price->customer_id}:{$price->variant_id}:{$startsAt}";

            if (!$validKeys->contains($key)) {
                $price->delete();
                $deletedCount++;
            }
        }

        $this->info("Cleanup completed: {$deletedCount} old price agreements removed.");

        Log::info('Price agreements cleanup completed', [
            'deleted_count' => $deletedCount,
            'processed_agreements' => $processedAgreements->count(),
        ]);
    }

    private function getConnector(Collection $connectors, string $connectorKey): null|BaseConnector|Cash
    {
        if ($connectors[$connectorKey]) {
            $class = $connectors[$connectorKey]->connector;

            return new $class;
        }

        return null;
    }
}
