<?php

namespace Daylight\ConnectorCash\Jobs;

use Daylight\ConnectorCash\Entities\Product as ProductEntity;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;

class UpdateProduct implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public ProductEntity $product,
    ) {
        //
    }

    public function handle(): ?string
    {
        // Get the variant and the product
        $variant = daylightModel('variant')::where('sku', $this->product->sku)->first();

        if ($variant) {
            // Restore the variant and the parent product if trashed
            if ($variant->trashed()) {
                $variant->restore();
                if ($variant->product()->trashed()) {
                    $variant->product()->restore();
                }
            }
        }

        $productClass = daylightModel('product');
        $product = $productClass::updateOrCreate(
            ['id' => $variant->product_id ?? null],
            [
                'brand_id' => null,
                'status' => $this->product->formatStatus(),
            ]
        );

        if (!$product instanceof $productClass) {
            $product = $productClass::find($product->id);
        }

        if (is_null($product)) {
            return false;
        }

        if ($this->product->title) {
            $translation = $product->translations()?->where([
                ['product_id', '=', $product->id],
                ['locale', '=', 'nl']
            ])?->first();

            if ($translation !== null) {
                $translation->updateQuietly([
                    'original_name' => $this->product->title,
                    'name' => $this->product->title
                ]);
            } else {
                $product->translations()->create(
                    [
                        'locale' => 'nl',
                        'slug' => Str::slug($this->product->title . '-' . $this->product->sku),
                        'name' => $this->product->title,
                        'original_name' => $this->product->title,
                        //'description' => $this->product->description,
                    ]
                );
            }
        }

        $variant = $product->variants()->updateOrCreate(
            [
                'sku' => $this->product->sku,
            ],
            [
                'sku' => $this->product->sku,
                'ean' => $this->product->ean,
                'articlegroup' => $this->product->articlegroup,
                'tax_class_id' => $this->product->getTaxClass(),
                'stock' => $this->product->formatQuantity(),
                'purchasable' => $this->product->canBePurchased(),
            ]
        );

        // Shipping
        $variant->shippingClasses()->syncWithoutDetaching(1);

//        $variant
//            ->prices()
//            ->whereCustomerGroupId(1)
//            ->updateOrCreate([
//                'price' => $this->product->sale_price,
//            ]);

        // Basic price
        $variant
            ->prices()
            ->updateOrCreate(
                [
                    'customer_group_id' => null,
                    'customer_id' => null,
                    'starts_at' => null,
                    'ends_at' => null,
                ],
                [
                    'price' => $this->product->sale_price,
                ]
            );

        return $this->product->sku;
    }

    public function uniqueId(): string
    {
        return $this->product->sku;
    }
}
