<?php

namespace Daylight\ConnectorCash\Jobs;

use App\Models\User;
use Daylight\Core\Models\Customer;
use Daylight\ConnectorCash\Entities\PriceList as PriceListEntity;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;
use Daylight\Core\Models\Enums\CustomerType;
use Daylight\Core\Models\Enums\AddressType;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class UpdatePriceList implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public PriceListEntity $priceList,
    )
    {
        //
    }

    public function handle(): void
    {
        if (!$this->priceList->list_name || $this->priceList->percentage == 0)  {
            Log::warning('PriceList has no list_name or is empty, skipping');
            return;
        }

        $customerGroup = daylightModel('customer_group')::firstOrCreate(
            ['name' => $this->priceList->list_name]
        );

        // Haal prefixes op
        $categoryPrefixes = DB::table('heg_category_structure')
            ->where('prefix', $this->priceList->category)
            ->orderByRaw('CHAR_LENGTH(prefix) DESC')
            ->pluck('prefix')
            ->toArray();

        if (empty($categoryPrefixes)) {
            Log::warning('No category structure found for category code', [
                'category_code' => $this->priceList->category,
            ]);
            return;
        }

        // UNION query voor alle prefixes
        $variantIdsQuery = collect($categoryPrefixes)
            ->map(function ($prefix) {
                return Variant::query()
                    ->where('sku', 'LIKE', $prefix . '%')
                    ->select('id');
            })
            ->reduce(function ($carry, $query) {
                return $carry ? $carry->union($query) : $query;
            })
            ->distinct();

        // Haal variant IDs op en SORTEER voor consistente lock volgorde
        $variantIds = DB::query()
            ->fromSub($variantIdsQuery, 'variant_ids')
            ->orderBy('id')
            ->pluck('id')
            ->toArray();

        if (empty($variantIds)) {
            Log::info('No variants found for category prefixes');
            return;
        }

        // Retry logic voor deadlock handling
        $maxRetries = 3;
        $retry = 0;

        while ($retry < $maxRetries) {
            try {
                $this->processPrices($variantIds, $customerGroup, $categoryPrefixes);
                break; //
            } catch (\Illuminate\Database\QueryException $e) {
                if ($e->getCode() == 40001 || str_contains($e->getMessage(), 'Deadlock')) {
                    $retry++;
                    if ($retry >= $maxRetries) {
                        Log::error('Price list processing failed after retries', [
                            'price_list_name' => $this->priceList->list_name,
                            'retries' => $retry,
                            'error' => $e->getMessage(),
                        ]);
                        throw $e;
                    }

                    usleep(50000 * $retry);
                    Log::warning('Deadlock detected, retrying', [
                        'attempt' => $retry,
                        'price_list_name' => $this->priceList->list_name,
                    ]);
                } else {
                    throw $e;
                }
            }
        }
    }

    private function processPrices(array $variantIds, $customerGroup, array $categoryPrefixes): void
    {
        DB::transaction(function () use ($variantIds, $customerGroup, $categoryPrefixes) {
            $customerGroup = daylightModel('customer_group')::lockForUpdate()
                ->find($customerGroup->id);

            // 1. Haal alle base prices op
            $basePrices = daylightModel('price')::query()
                ->whereIn('variant_id', $variantIds)
                ->whereNull('customer_group_id')
                ->whereNull('customer_id')
                ->where('min_quantity', '<=', 1)
                ->get()
                ->keyBy('variant_id');

            // 2. Haal bestaande prices op voor deze customer group
            $existingPrices = daylightModel('price')::query()
                ->whereIn('variant_id', $variantIds)
                ->where('customer_group_id', $customerGroup->id)
                ->whereNull('customer_id')
                ->where('min_quantity', 1)
                ->get()
                ->keyBy('variant_id');

            // 3. Bereid upsert data voor
            $startsAt = $this->priceList->date
                ? Carbon::parse($this->priceList->date)->startOfDay()
                : null;

            $upsertData = [];
            $now = now();
            $updatedVariantIds = [];

            foreach ($variantIds as $variantId) {
                $basePrice = $basePrices->get($variantId);

                if (!$basePrice) {
                    continue;
                }

                $basePriceValue = $basePrice->getRawOriginal('price');
                $discountMultiplier = 1 - ($this->priceList->percentage / 100);
                $newPriceValue = (int) round($basePriceValue * $discountMultiplier);

                $existingPrice = $existingPrices->get($variantId);
                $priceId = $existingPrice?->id;

                $upsertData[] = [
                    'id' => $priceId, // NULL voor nieuwe records
                    'variant_id' => $variantId,
                    'customer_group_id' => $customerGroup->id,
                    'customer_id' => null,
                    'price' => $newPriceValue,
                    'min_quantity' => 1,
                    'starts_at' => $startsAt,
                    'ends_at' => null,
                    'created_at' => $priceId ? $existingPrice->created_at : $now,
                    'updated_at' => $now,
                ];

                $updatedVariantIds[] = $variantId;
            }

            // 4. Upsert in chunks (gebruikt INSERT ... ON DUPLICATE KEY UPDATE)
            if (!empty($upsertData)) {
                collect($upsertData)->chunk(500)->each(function ($chunk) {
                    daylightModel('price')::query()->upsert(
                        $chunk->toArray(),
                        ['variant_id', 'customer_group_id', 'customer_id', 'min_quantity'], // unique columns
                        ['price', 'starts_at', 'ends_at', 'updated_at'] // columns to update
                    );
                });
            }

            // 5. Verwijder prices die niet meer in de prijslijst zitten
            // (varianten die wel een price hadden maar niet in $updatedVariantIds zitten)
            $variantIdsToDelete = $existingPrices->keys()
                ->diff($updatedVariantIds)
                ->toArray();

            if (!empty($variantIdsToDelete)) {
                daylightModel('price')::query()
                    ->whereIn('variant_id', $variantIdsToDelete)
                    ->where('customer_group_id', $customerGroup->id)
                    ->whereNull('customer_id')
                    ->where('min_quantity', 1)
                    ->delete();
            }

            Log::info('Price list processed', [
                'price_list_name' => $this->priceList->list_name,
                'customer_group_id' => $customerGroup->id,
                'variants_processed' => count($upsertData),
                'variants_total' => count($variantIds),
                'variants_deleted' => count($variantIdsToDelete),
            ]);
        });
    }

//    private function processPrices(array $variantIds, $customerGroup, array $categoryPrefixes): void
//    {
//        DB::transaction(function () use ($variantIds, $customerGroup, $categoryPrefixes) {
//            $customerGroup = daylightModel('customer_group')::lockForUpdate()
//                ->find($customerGroup->id);
//
//            // 1. Haal alle base prices op in één query
//            $basePrices = daylightModel('price')::query()
//                ->whereIn('variant_id', $variantIds)
//                ->whereNull('customer_group_id')
//                ->whereNull('customer_id')
//                ->where('min_quantity', '<=', 1)
//                ->get()
//                ->keyBy('variant_id');
//
//            // 2. Bulk delete - gebruik IN met gesorteerde IDs
//            daylightModel('price')::query()
//                ->whereIn('variant_id', $variantIds)
//                ->where('customer_group_id', $customerGroup->id)
//                ->delete();
//
//            // 3. Bereid alle nieuwe prices voor
//            $startsAt = $this->priceList->date
//                ? Carbon::parse($this->priceList->date)->startOfDay()
//                : null;
//
//            $newPrices = [];
//            $now = now();
//
//            foreach ($variantIds as $variantId) {
//                $basePrice = $basePrices->get($variantId);
//
//                if (!$basePrice) {
//                    continue;
//                }
//
//                $basePriceValue = $basePrice->getRawOriginal('price');
//                $discountMultiplier = 1 - ($this->priceList->percentage / 100);
//                $newPriceValue = (int) round($basePriceValue * $discountMultiplier);
//
//                $newPrices[] = [
//                    'variant_id' => $variantId,
//                    'customer_group_id' => $customerGroup->id,
//                    'customer_id' => null,
//                    'price' => $newPriceValue,
//                    'min_quantity' => 1,
//                    'starts_at' => $startsAt,
//                    'ends_at' => null,
//                    'created_at' => $now,
//                    'updated_at' => $now,
//                ];
//            }
//
//            // 4. Bulk insert in chunks
//            if (!empty($newPrices)) {
//                collect($newPrices)->chunk(500)->each(function ($chunk) {
//                    daylightModel('price')::query()->insert($chunk->toArray());
//                });
//            }
//
//            Log::info('Price list processed', [
//                'price_list_name' => $this->priceList->list_name,
//                'customer_group_id' => $customerGroup->id,
//                'variants_processed' => count($newPrices),
//                'variants_total' => count($variantIds),
//            ]);
//        });
//    }

}
