<?php

namespace Daylight\ConnectorCash\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\ConnectorCash\Http\Cash;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Connector\Support\ConnectorLogger;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\ConnectorCash\Jobs\UpdateProduct;
use Illuminate\Support\Facades\Log;
use Illuminate\Bus\PendingBatch;
use Daylight\Core\Models\Product;

// php artisan connector:update-products --connector=cash --force

class UpdateProducts extends Command
{
    protected $signature = 'connector:update-products {--connector=} {--startfrom=} {--force}';

    protected $description = 'Get Products from the ERP';

    protected BaseConnector|Cash $connector;

    public function handle(ConnectorRegistry $registry): int
    {
        $iteration = 1;
        $lastItem = $this->option('startfrom') ?? 1;

        $this->connector = $this->getConnector(
            $registry->all(),
            $this->option('connector') ?? ''
        );

        if (!$this->connector) {
            $this->error('No active connector found.');
            return Command::FAILURE;
        }

        if (!method_exists($this->connector, 'getProductMapper')) {
            $this->error('No products mapper found in ' .  $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start import for ' . $this->connector->getName() . ', ' . $lastItem);

        LazyCollection::make(function () use (&$iteration, &$lastItem) {
            $prevLastItem = null;

//            do {
            while (true) {
                $this->line('update-products startfrom=' . $lastItem);

                try {
                    // Cash requires the last loaded item to start the next 10000 items
                    $results = $this->getItems($lastItem);
                } catch (\Exception $e) {
                    $this->error('Exception ' . $e->getMessage());
                    $results = collect();
                }

                if ($results->isEmpty()) {
                    $this->line('-> data: 0, stop');
                    break;
                }

                $last = $results->last();
                $lastSku = is_object($last) ? ($last->sku ?? null) : (is_array($last) ? ($last['sku'] ?? null) : null);

                $this->line('-> data: ' . $results->count() . ', lastSku=' . var_export($lastSku, true) . ', it=' . $iteration . ', lastItem=' . $lastItem );

                if ($lastSku === null) {
                    $this->warn('Kan laatste sku niet lezen; eerste item: ' . json_encode($results->first()));
                    break;
                }

                yield $results;

                $lastItem = (int)$lastSku;

                if ($lastItem === $prevLastItem) {
                    $this->error('Pagination key did not advance (' . $lastItem . '). Stop.');
                    break;
                }
                $prevLastItem = $lastItem;

                $iteration++;
            }
//           } while ($results->isNotEmpty());
        })->each(function (Collection $products, $index) {
            $jobs = [];

            foreach ($products as $product) {
                $jobs[] = new UpdateProduct($product);
            }

            Bus::batch($jobs)
                ->name('cash:products:iteration:' . ($index + 1))
                ->onQueue($this->connector->getKey())
                ->dispatch();

            unset($jobs, $products);
            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        return Command::SUCCESS;
    }

    protected function getItems(int $page = 1, int $perPage = 10000): ?Collection
    {
        return $this->connector
            ->products()
            ->all($this->connector->getProductMapper(), ['pageNumber' => $page, 'perPage' => $perPage])
            ->dtoOrFail();
    }

    private function getConnector(Collection $connectors, string $connectorKey): null|BaseConnector|Cash
    {
        if ($connectors[$connectorKey]) {
            $class = $connectors[$connectorKey]->connector;

            return new $class;
        }

        return null;
    }
}
