<?php

namespace Daylight\ConnectorCash\Jobs;

use Daylight\Core\Models\Product;
use Daylight\ConnectorCash\Entities\Product as ProductEntity;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;

class UpdateProduct implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public ProductEntity $product,
    ) {
        //
    }

    public function handle(): ?string
    {
        // Get the variant and the product
        $variant = Variant::where('sku', $this->product->sku)->first();

        if ($variant) {
            // Restore the variant and the parent product if trashed
            if ($variant->trashed()) {
                $variant->restore();
                if ($variant->product()->trashed()) {
                    $variant->product()->restore();
                }
            }
        }

        $product = Product::updateOrCreate(
            ['id' => $variant->product_id ?? null],
            [
                'brand_id' => null,
                'status' => $this->product->formatStatus(),
            ]
        );

        if (is_null($product)) {
            return false;
        }

        if ($this->product->description) {
            $translation = $product->translations()?->where([
                ['product_id', '=', $product->id],
                ['locale', '=', 'nl']
            ])?->first();

            if ($translation !== null) {
                $translation->description = $this->product->description;
                $translation->name = $this->product->title;
                $translation->save();
            } else {
                $product->translations()->create(
                    [
                        'locale' => 'nl',
                        'slug' => Str::slug($this->product->title . '-' . $this->product->sku),
                        'name' => $this->product->description,
                        'description' => $this->product->description,
                    ]
                );
            }
        }

        $variant = $product->variants()->updateOrCreate(
            [
                'sku' => $this->product->sku,
            ],
            [
                'sku' => $this->product->sku,
                'ean' => $this->product->ean,
//                'title' => $this->product->title,
//                'description' => $this->product->description,
                'tax_class_id' => $this->product->getTaxClass(),
                'stock' => $this->product->formatQuantity(),
                'purchasable' => $this->product->canBePurchased(),
//                'weight' => $this->product->specifications?->netWeight,
//                'length' => $this->product->specifications?->length,
//                'width' => $this->product->specifications?->width,
//                'height' => $this->product->specifications?->height,
//                'diameter' => $this->product->specifications?->diameter,
//                'volume' => $this->product->specifications?->volume,
            ]
        );

        // Shipping
        $variant->shippingClasses()->attach(1);

        $variant
            ->prices()
            ->whereCustomerGroupId(1)
            ->updateOrCreate([
                'price' => $this->product->sale_price,
            ]);

//        UpdateArticleImages::dispatchIf(
//            $this->product->canBePurchased(),
//            article: $this->product,
//            variant: $variant
//        )->onQueue('bytescale');

//        Product::updateOrCreate([
//            'imdex_id' => $this->product->articleNumber,
//        ], [
//            'data' => $this->product->toJson(),
//        ]);

        return $this->product->sku;
    }

    public function uniqueId(): string
    {
        return $this->product->sku;
    }
}
