<?php

namespace Daylight\ConnectorCash\Console;

use Daylight\Connector\Services\BaseConnector;
use Daylight\ConnectorCash\Http\Cash;
use Illuminate\Console\Command;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Connector\Support\ConnectorLogger;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Facades\Bus;
use Daylight\ConnectorCash\Jobs\UpdateOrder;
use Illuminate\Support\Facades\Log;
use Illuminate\Bus\PendingBatch;
use Daylight\Core\Models\Product;

class UpdateOrders extends Command
{
    protected $signature = 'connector:update-orders {--connector=} {--orderNumber=} {--force}';

    protected $description = 'Get Orders from the ERP';

    protected BaseConnector|Cash $connector;

    public function handle(ConnectorRegistry $registry): int
    {
        $iteration = 1;

        $this->connector = $this->getConnector(
            $registry->all(),
            $this->option('connector') ?? ''
        );

        if (!$this->connector) {
            $this->error('No active connector found.');
            return Command::FAILURE;
        }

        if (!method_exists($this->connector, 'getOrderMapper')) {
            $this->error('No orders mapper found in ' . $this->connector->getName());
            return Command::FAILURE;
        }

        $this->line('Start import for ' . $this->connector->getName());

        LazyCollection::make(function () use (&$iteration) {
            try {
                $results = $this->getItems();
            } catch (\Exception $e) {
                $this->error('Exception ' . $e->getMessage());
                $results = collect();
            }

            if (count($results) === 0) {
                return;
            }

            yield $results;

            $iteration++;
        })->each(function (Collection $orders, $index) {
            $jobs = [];

            foreach ($orders as $order) {
                if (!$order->items) {
                    $order->items = $this->getOrderItems($order->external_id);
                }

                $jobs[] = new UpdateOrder($order);
            }

            Bus::batch($jobs)
                ->name('cash:orders:iteration:' . ($index + 1))
                ->onQueue($this->connector->getKey())
                ->dispatch();

            unset($jobs);
            unset($orders);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page ' . ($index + 1));
        });

        return Command::SUCCESS;
    }

    protected function getItems(int $page = 1, int $perPage = 250): ?Collection
    {
        return $this->connector
            ->orders()
            ->all($this->connector->getOrderMapper(), ['pageNumber' => $page, 'perPage' => $perPage, 'orderNumber' => $this->option('orderNumber') ?? null])
            ->dtoOrFail();
    }

    protected function getOrderItems(string $orderNumber)
    {
        return $this->connector
            ->orderItems()
            ->get($this->connector->getOrderItemMapper(), ['orderNumber' => $orderNumber])
            ->dtoOrFail();
    }

    private function getConnector(Collection $connectors, string $connectorKey): null|BaseConnector|Cash
    {
        if ($connectors[$connectorKey]) {
            $class = $connectors[$connectorKey]->connector;

            return new $class;
        }

        return null;
    }
}
