<?php

namespace Daylight\Connector\Exact\Entities;

use Carbon\Carbon;
use Daylight\Connector\Exact\Entities\Casts\TimestampDateCast;
use Daylight\Core\Modules\Multilingualism\Multilingualism;
use Spatie\LaravelData\Attributes\MapInputName;
use Spatie\LaravelData\Attributes\WithCast;
use Spatie\LaravelData\Data;

class Contact extends Data
{
    public function __construct(
        #[MapInputName('ID')]
        public string $id,

        #[MapInputName('Timestamp')]
        public ?int $timestamp,

        #[MapInputName('Account')]
        public string $accountId,

        #[MapInputName('Email')]
        public ?string $email = null,

        #[MapInputName('Mobile')]
        public ?string $phone = null,

        #[MapInputName('FirstName')]
        public ?string $firstName = null,

        #[MapInputName('MiddleName')]
        public ?string $middleName = null,

        #[MapInputName('LastName')]
        public ?string $lastName = null,

        #[MapInputName('Language')]
        public ?string $locale = 'nl',

        #[MapInputName('AccountIsCustomer')]
        public bool $isCustomer = false,

        #[MapInputName('IsMainContact')]
        public bool $mainContact = false,

        #[MapInputName('IsAnonymised')]
        public bool $isAnonymised = false,

        #[MapInputName('StartDate')]
        #[WithCast(TimestampDateCast::class)]
        public ?Carbon $startDate = null,

        #[MapInputName('EndDate')]
        #[WithCast(TimestampDateCast::class)]
        public ?Carbon $endDate = null,
    ) {
        //
    }

    public function shouldBeImported(): bool
    {
        return $this->isCustomer
            && ! $this->mainContact
            && ! $this->isAnonymised
            && ! is_null($this->email)
            && ($this->startDate === null || $this->startDate->lessThanOrEqualTo(now()))
            && ($this->endDate === null || $this->endDate->greaterThanOrEqualTo(now()));
    }

    public function getFirstName(): ?string
    {
        return $this->firstName;
    }

    public function getLastName(): ?string
    {
        return $this->middleName ? trim("{$this->middleName} {$this->lastName}") : $this->lastName;
    }

    public function getLocale(): string
    {
        $locale = strtolower(trim($this->locale));
        $locales = Multilingualism::getAvailableLocales();

        if ($locale && $locales->contains($locale)) {
            return $locale;
        }

        return config('multilingualism.default_locale');
    }
}
