<?php

namespace Daylight\Connector\Exact;

use Daylight\Connector\Exact\Models\OAuthToken;
use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Facades\Cache;
use Saloon\Http\Auth\AccessTokenAuthenticator;

class TokenStorage
{
    public const string PROVIDER = 'exact';

    private const string CACHE_KEY = 'exact:oauth:authenticator';

    private const string LOCK_KEY = 'exact:oauth:refresh';

    private const int CACHE_TAIL_SECONDS = 30;

    private const int LOCK_TTL_SECONDS = 10;

    private const int LOCK_WAIT_SECONDS = 5;

    public function __construct(
        private readonly Exact $connector,
    ) {
        //
    }

    public function persistInitialAuthenticator(AccessTokenAuthenticator $auth): void
    {
        OAuthToken::updateOrCreate(
            ['provider' => self::PROVIDER],
            ['authenticator' => $auth]
        );

        Cache::forget(self::CACHE_KEY);
    }

    public function getAuthenticator(): AccessTokenAuthenticator
    {
        if ($cached = Cache::get(self::CACHE_KEY)) {
            $auth = $cached;

            if ($auth->hasNotExpired()) {
                return $auth;
            }
        }

        $row = OAuthToken::where('provider', self::PROVIDER)->firstOrFail();

        $auth = $row->authenticator;

        if ($auth->hasNotExpired()) {
            $this->putInCache($auth);

            return $auth;
        }

        $lock = Cache::lock(self::LOCK_KEY, self::LOCK_TTL_SECONDS);

        return $lock->block(self::LOCK_WAIT_SECONDS, function () use ($row): AccessTokenAuthenticator {
            $freshRow = $row->fresh();
            $current = $freshRow->authenticator;

            if ($current->hasNotExpired()) {
                $this->putInCache($current);

                return $current;
            }

            $refreshed = $this->connector->refreshAccessToken($current);

            $freshRow->update(['authenticator' => $refreshed]);

            $this->putInCache($refreshed);

            return $refreshed;
        });
    }

    public function getAccessToken(): string
    {
        return $this->getAuthenticator()->getAccessToken();
    }

    public function withAuthenticator(callable $fn)
    {
        try {
            return $fn($this->getAuthenticator());
        } catch (RequestException $e) {
            if ($e->response && $e->response->status() === 401) {
                $this->invalidateCache();

                return $fn($this->getAuthenticator());
            }

            throw $e;
        }
    }

    public function withToken(callable $fn)
    {
        return $this->withAuthenticator(fn (AccessTokenAuthenticator $auth) => $fn($auth->getAccessToken()));
    }

    public function invalidateCache(): void
    {
        Cache::forget(self::CACHE_KEY);
    }

    public function forceRefresh(): AccessTokenAuthenticator
    {
        $row = OAuthToken::where('provider', self::PROVIDER)->firstOrFail();

        $refreshed = $this->connector->refreshAccessToken(
            refreshToken: $row->authenticator
        );

        $row->update([
            'authenticator' => $refreshed,
        ]);

        $this->putInCache($refreshed);

        return $refreshed;
    }

    private function putInCache(AccessTokenAuthenticator $auth): void
    {
        $expiry = $auth->getExpiresAt();

        Cache::put(
            key: self::CACHE_KEY,
            value: $auth,
            ttl: max(5, $expiry->getTimestamp() - time() - self::CACHE_TAIL_SECONDS)
        );
    }
}
