<?php

namespace Daylight\Connector\Exact\Console\Commands;

use Daylight\Connector\Exact\Entities\Customer;
use Daylight\Connector\Exact\Entities\PaginatedResource;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateCustomer;
use Daylight\Connector\Exact\Models\Import;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\LazyCollection;

class UpdateCustomers extends Command
{
    protected $signature = 'exact:update-customers {--debug : Show debug information} {--force : Force update even if no changes}';

    protected $description = 'Update customers from Exact Online';

    protected int $totalRows = 0;

    public function handle(): int
    {
        $this->info('Updating customers...');

        $skipToken = null;
        $currentCursor = $this->option('force') ? 1 : $this->getCurrentCursor();
        $cursor = max($currentCursor, 0);

        $this->logMemoryUsage($this->option('debug'));

        $this->line('Fetching items starting from cursor: '.$currentCursor);

        LazyCollection::make(function () use (&$skipToken, &$currentCursor) {
            do {
                try {
                    $results = $this->getItems($currentCursor, $skipToken);
                } catch (\Exception $e) {
                    $results = null;
                }

                if (is_null($results) || $results->getData()->isEmpty()) {
                    return;
                }

                $skipToken = $results->getSkipToken();

                yield $results;

            } while ($results->hasMore());
        })->each(function (PaginatedResource $items, $index) use (&$cursor) {
            sleep(2);

            $this->info('Processing page '.($index + 1).' with '.$items->getData()->count().' items.');

            $this->totalRows += $items->getData()->count();

            $jobs = $items->getData()->map(function (Customer $customer) use (&$cursor) {
                $cursor = max($cursor, $customer->timestamp);

                return new UpdateCustomer(
                    customer: $customer
                );
            });

            Bus::batch($jobs)
                ->name('exact:customers:iteration:'.($index + 1))
                ->onQueue('exact')
                ->dispatch();

            unset($jobs);
            unset($items);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page '.($index + 1));

            $this->logMemoryUsage($this->option('debug'));
        });

        Import::updateOrCreate([
            'type' => 'customers',
        ], [
            'cursor' => $cursor,
        ]);

        $this->logMemoryUsage($this->option('debug'));

        $this->info('Total items processed: '.$this->totalRows);

        return Command::SUCCESS;
    }

    protected function getCurrentCursor(): int
    {
        return Import::query()
            ->firstWhere('type', 'customers')
            ?->cursor ?? 0;
    }

    protected function getItems(int $cursor, ?string $skipToken = null): ?PaginatedResource
    {
        return $this->connector()
            ->customers()
            ->all(
                mapper: $this->connector()->getCustomerMapper(),
                options: [
                    'cursor' => $cursor,
                    'skiptoken' => $skipToken,
                ]
            )
            ->dto();
    }

    private function connector(): Exact
    {
        return app(Exact::class);
    }

    private function logMemoryUsage(bool $debug = false): void
    {
        if (! $debug) {
            return;
        }

        $this->line('--> Memory Usage: '.round(memory_get_usage() / 1024 / 1024, 2).' MB');
        $this->line('--> Peak Memory Usage: '.round(memory_get_peak_usage() / 1024 / 1024, 2).' MB');
        $this->newLine();
    }
}
