<?php

namespace Daylight\Connector\Exact\Jobs;

use Daylight\Connector\Exact\Entities\Customer;
use Daylight\Core\Models\Country;
use Daylight\Core\Models\CustomerGroup;
use Daylight\Core\Models\Enums\CustomerType;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class UpdateCustomer implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public Customer $customer
    ) {
        //
    }

    public function handle(): void
    {
        if (! $this->customer->shouldBeImported()) {
            return;
        }

        $customer = daylightModel('customer')::query()
            ->updateOrCreate([
                'external_id' => $this->customer->id,
            ], [
                'type' => CustomerType::COMPANY->value,
                'name' => $this->customer->name,
                'vat_number' => $this->customer->vatNumber,
                'status' => $this->customer->isActive() ? 1 : 0,
                'address_line_1' => $this->customer->addressLine1,
                'address_line_2' => $this->customer->addressLine2,
                'postal_code' => $this->customer->postalCode,
                'city' => $this->customer->city,
                'country_id' => $this->getCountry()?->id,
                'invoice_payments' => $this->customer->canPayOnInvoice() ? 1 : 0,
                'package_delivery' => $this->customer->isPackageDelivery() ? 1 : 0,
            ]);

        $customerGroup = $this->getCustomerGroup();

        if ($customerGroup) {
            $customer->customerGroups()->attach($customerGroup);
        }
    }

    public function getCustomerGroup(): ?CustomerGroup
    {
        return daylightModel('customer_group')::query()
            ->whereExternalId($this->customer->priceListId)
            ->first();
    }

    public function getCountry(): ?Country
    {
        return daylightModel('country')::query()
            ->whereCode(trim($this->customer->country))
            ->first();
    }
}
