<?php

namespace Daylight\Connector\Exact\Jobs;

use Daylight\Connector\Exact\Entities\Address;
use Daylight\Core\Models\Country;
use Daylight\Core\Models\Customer;
use Daylight\Core\Services\Countries;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class UpdateAddress implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public Address $address
    ) {
        //
    }

    public function handle(): void
    {
        $customer = $this->getCustomer();
        $country = $this->getCountry();

        if (! $customer || ! $country) {
            return;
        }

        if (! $this->address->addressLine1 || ! $this->address->postalCode || ! $this->address->city) {
            return;
        }

        $customer->addresses()->updateOrCreate([
            'external_id' => $this->address->id,
        ], [
            'type' => $this->address->getType(),
            'default' => $this->address->isDefault ? 1 : 0,
            'name' => $this->address->name,
            'address_line_1' => $this->address->addressLine1,
            'address_line_2' => $this->address->addressLine2,
            'postal_code' => $this->address->postalCode,
            'city' => $this->address->city,
            'country_id' => $country->id,
            'vat_number' => $customer->vat_number,
            'vat_number_verified_at' => $customer->vat_number && $country->id !== Countries::defaultCountry()->id ? now() : null,
        ]);
    }

    public function getCustomer(): ?Customer
    {
        if (! $this->address->accountId) {
            return null;
        }

        return daylightModel('customer')::query()
            ->whereExternalId($this->address->accountId)
            ->first();
    }

    public function getCountry(): ?Country
    {
        return daylightModel('country')::query()
            ->whereCode(trim($this->address->country))
            ->first();
    }
}
