<?php

namespace Daylight\Connector\Exact\Jobs;

use App\Models\User;
use Daylight\Connector\Exact\Entities\Deletion;
use Daylight\Connector\Exact\Entities\ProductPrice;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class DeleteRow implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public Deletion $deletion
    ) {
        //
    }

    public function handle(): void
    {
        $type = $this->getType();

        if (method_exists($this, 'delete'.$type) && ! is_null($type)) {
            $method = 'delete'.$type;

            $this->{$method}();
        }
    }

    public function deleteAccounts(): void
    {
        $customer = daylightModel('customer')::query()
            ->where('external_id', $this->deletion->entityKey)
            ->first();

        if (! $customer) {
            return;
        }

        $customer->delete();
    }

    public function deleteAddresses(): void
    {
        $address = daylightModel('address')::query()
            ->where('external_id', $this->deletion->entityKey)
            ->first();

        if (! $address) {
            return;
        }

        $address->delete();
    }

    public function deleteContacts(): void
    {
        $user = User::query()
            ->where('external_id', $this->deletion->entityKey)
            ->first();

        if (! $user) {
            return;
        }

        $user->delete();
    }

    public function deleteItemPrices(): void
    {
        $itemPrice = daylightModel('price')::query()
            ->whereSource(ProductPrice::class)
            ->whereExternalId($this->deletion->entityKey)
            ->first();

        if (! $itemPrice) {
            return;
        }

        $itemPrice->delete();
    }

    public function deleteItems(): void
    {
        $item = daylightModel('variant')::query()
            ->whereExternalId($this->deletion->entityKey)
            ->first();

        if (! $item) {
            return;
        }

        $item->delete();
    }

    public function deletePaymentTerms(): void
    {
        Log::info('PaymentTerms deletion not implemented yet for entityKey: '.$this->deletion->entityKey);
    }

    public function deleteSalesInvoices(): void
    {
        Log::info('SalesInvoices deletion not implemented yet for entityKey: '.$this->deletion->entityKey);
    }

    public function getType(): ?string
    {
        return match ($this->deletion->entityType) {
            //            1 => 'TransactionLines',
            2 => 'Accounts',
            3 => 'Addresses',
            //            4 => 'Attachments',
            5 => 'Contacts',
            //            6 => 'Documents',
            //            7 => 'GLAccounts',
            8 => 'ItemPrices',
            9 => 'Items',
            10 => 'PaymentTerms',
            //            12 => 'SalesOrders', // This entity is going to be removed. Please refer to SalesOrderHeaders, SalesOrderLines.
            13 => 'SalesInvoices',
            //            14 => 'TimeCostTransactions',
            //            15 => 'StockPositions',
            //            16 => 'GoodsDeliveries',
            //            17 => 'GoodsDeliveryLines',
            //            18 => 'GLClassifications',
            //            19 => 'ItemWarehouses',
            //            20 => 'StorageLocationStockPositions',
            //            21 => 'Projects',
            //            22 => 'PurchaseOrders',
            //            23 => 'Subscriptions',
            //            24 => 'SubscriptionLines',
            //            25 => 'ProjectWBS',
            //            26 => 'ProjectPlanning',
            //            27 => 'LeaveAbsenceHoursByDay',
            //            28 => 'SerialBatchNumbers',
            //            29 => 'StockSerialBatchNumbers',
            30 => 'ItemAccounts',
            31 => 'DiscountTables',
            32 => 'SalesOrderHeaders',
            33 => 'SalesOrderLines',
            34 => 'QuotationHeaders',
            35 => 'QuotationLines',
            //            36 => 'ShopOrders',
            //            37 => 'ShopOrderMaterialPlans',
            //            38 => 'ShopOrderRoutingStepPlans',
            //            39 => 'Schedules',
            //            40 => 'ScheduleEntries',
            //            41 => 'ItemStorageLocations',
            //            42 => 'Employees',
            //            43 => 'Employments',
            //            44 => 'EmploymentContracts',
            //            45 => 'EmploymentOrganizations',
            //            46 => 'EmploymentCLAs',
            //            47 => 'EmploymentSalaries',
            //            48 => 'BankAccounts',
            //            49 => 'EmploymentTaxAuthoritiesGeneral',
            //            50 => 'ShopOrderPurchasePlanning',
            //            51 => 'ShopOrderSubOrders',
            //            53 => 'RequirementIssues',
            //            54 => 'BillOfMaterialMaterials',
            //            55 => 'BillOfMaterialVersions',
            //            56 => 'LeaveRegistrations',
            //            57 => 'LeaveBuildUpRegistrations',
            //            58 => 'AbsenceRegistrationTransactions',
            //            59 => 'AbsenceRegistrations',
            default => null
        };
    }
}
