<?php

namespace Daylight\Connector\Exact\Console\Commands;

use Daylight\Connector\Exact\Entities\ProductExtraField;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateProductPackaging;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Bus;

class UpdateArticlePackaging extends Command
{
    protected $signature = 'exact:update-article-packaging {--debug : Show debug information}';

    protected $description = 'Update article packaging from Exact Online';

    protected int $totalRows = 0;

    public function handle(): int
    {
        $this->info('Updating article packaging...');

        $this->logMemoryUsage($this->option('debug'));

        $jobs = $this->getItems()->map(function (ProductExtraField $item) {
            return new UpdateProductPackaging($item);
        });

        $this->logMemoryUsage($this->option('debug'));

        Bus::batch($jobs)
            ->name('exact:product-packaging')
            ->onQueue('exact')
            ->dispatch();

        $this->logMemoryUsage($this->option('debug'));

        $this->info('Total items processed: '.$jobs->count());

        return Command::SUCCESS;
    }

    protected function getItems(): ?Collection
    {
        return $this->connector()
            ->products()
            ->extraFields(
                mapper: $this->connector()->getProductExtraFieldMapper(),
                options: [
                    'number' => 1,
                ]
            )
            ->dto();
    }

    private function connector(): Exact
    {
        return app(Exact::class);
    }

    private function logMemoryUsage(bool $debug = false): void
    {
        if (! $debug) {
            return;
        }

        $this->line('--> Memory Usage: '.round(memory_get_usage() / 1024 / 1024, 2).' MB');
        $this->line('--> Peak Memory Usage: '.round(memory_get_peak_usage() / 1024 / 1024, 2).' MB');
        $this->newLine();
    }
}
