<?php

namespace Daylight\Connector\Exact\Console\Commands;

use Daylight\Connector\Exact\Entities\Order;
use Daylight\Connector\Exact\Entities\PaginatedResource;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateOrder;
use Daylight\Connector\Exact\Models\Import;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\LazyCollection;

class UpdateOrders extends Command
{
    protected $signature = 'exact:update-orders {--debug : Show debug information}';

    protected $description = 'Update orders from Exact Online';

    protected int $totalRows = 0;

    public function handle(): int
    {
        $this->info('Updating orders...');

        $skipToken = null;

        //        $results = $this->getItems($skipToken);
        //        UpdateOrder::dispatchSync($results->getData()->first());
        //        die;

        $this->logMemoryUsage($this->option('debug'));

        LazyCollection::make(function () use (&$skipToken, &$currentCursor) {
            do {
                $this->line('Fetching items with skip token: '.($skipToken ?? 'none'));

                try {
                    $results = $this->getItems($skipToken);
                } catch (\Exception $e) {
                    $results = null;
                }

                if (is_null($results) || $results->getData()->isEmpty()) {
                    return;
                }

                $skipToken = $results->getSkipToken();

                yield $results;

            } while ($results->hasMore());
        })->each(function (PaginatedResource $items, $index) use (&$cursor) {
            sleep(2);

            $this->info('Processing page '.($index + 1).' with '.$items->getData()->count().' items.');

            $this->totalRows += $items->getData()->count();

            $jobs = $items->getData()->map(function (Order $item) {
                return new UpdateOrder(
                    order: $item
                );
            });

            Bus::batch($jobs)
                ->name('exact:orders:iteration:'.($index + 1))
                ->onQueue('exact')
                ->dispatch();

            unset($jobs);
            unset($items);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page '.($index + 1));

            $this->logMemoryUsage($this->option('debug'));
        });

        $this->logMemoryUsage($this->option('debug'));

        $this->info('Total items processed: '.$this->totalRows);

        return Command::SUCCESS;
    }

    protected function getCurrentCursor(): int
    {
        return Import::query()
            ->firstWhere('type', 'orders')
            ?->cursor ?? 0;
    }

    protected function getItems(?string $skipToken = null): ?PaginatedResource
    {
        return $this->connector()
            ->orders()
            ->all(
                mapper: $this->connector()->getOrderMapper(),
                options: [
                    'skiptoken' => $skipToken,
                ]
            )
            ->dto();
    }

    private function connector(): Exact
    {
        return app(Exact::class);
    }

    private function logMemoryUsage(bool $debug = false): void
    {
        if (! $debug) {
            return;
        }

        $this->line('--> Memory Usage: '.round(memory_get_usage() / 1024 / 1024, 2).' MB');
        $this->line('--> Peak Memory Usage: '.round(memory_get_peak_usage() / 1024 / 1024, 2).' MB');
        $this->newLine();
    }
}
