<?php

namespace Daylight\Connector\Exact\Jobs;

use Daylight\Connector\Exact\Entities\PricingListLine;
use Daylight\Connector\Exact\Entities\ProductPrice;
use Daylight\Connector\Exact\Models\CustomerGroupPeriod;
use Daylight\Core\Models\CustomerGroup;
use Daylight\Core\Models\Price;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;

class UpdatePricingListLine implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public PricingListLine $pricingListLine
    ) {
        //
    }

    public function handle(): void
    {
        $customerGroup = $this->getCustomerGroup();
        $variants = $this->getVariants();

        if (! $customerGroup || $variants->isEmpty()) {
            return;
        }

        $variants->each(function (Variant $variant) use ($customerGroup) {
            try {
                $variant->prices()->updateOrCreate([
                    'customer_group_id' => $customerGroup->id,
                    'customer_id' => null,
                    'external_id' => $this->pricingListLine->id,
                    'source' => PricingListLine::class,
                ], [
                    'price' => $this->calculatePrice($variant),
                    'min_quantity' => $this->pricingListLine->quantity,
                    'starts_at' => $this->getCustomerGroupPeriod()?->starts_at,
                    'ends_at' => $this->getCustomerGroupPeriod()?->ends_at,
                ]);
            } catch (\Exception $e) {
                $this->fail('Failed to update prices for price list line ['.$this->pricingListLine->id.']');
            }
        });
    }

    public function calculatePrice(Variant $variant): float
    {
        if ($this->pricingListLine->discountMethod === 2) {
            return $this->pricingListLine->price;
        }

        if ($this->pricingListLine->basePriceAmount) {
            return (1 - $this->pricingListLine->discountMultiplier) * $this->pricingListLine->basePriceAmount;
        }

        if ($this->pricingListLine->basePriceId) {
            $basePrice = $this->getBasePrice($this->pricingListLine->basePriceId);
        } else {
            $basePrice = $variant->prices
                ->whereNull('customer_group_id')
                ->whereNull('customer_id')
                ->firstOrFail();
        }

        return (1 - $this->pricingListLine->discountMultiplier) * $basePrice->price;
    }

    public function getBasePrice(string $priceId): Price
    {
        return Price::query()
            ->where(function ($builder) use ($priceId) {
                $builder
                    ->whereExternalId($priceId)
                    ->whereSource(ProductPrice::class);
            })
            ->firstOrFail();
    }

    public function getVariants(): Collection
    {
        return Variant::query()
            ->with('prices')
            ->when($this->pricingListLine->productId, fn ($query) => $query->where('external_id', $this->pricingListLine->productId))
            ->when($this->pricingListLine->productGroupId, fn ($query) => $query->where('external_product_group_id', $this->pricingListLine->productGroupId))
            ->get();
    }

    public function getCustomerGroup(): ?CustomerGroup
    {
        return CustomerGroup::query()
            ->whereCode($this->pricingListLine->priceListCode)
            ->first();
    }

    public function getCustomerGroupPeriod(): ?CustomerGroupPeriod
    {
        return CustomerGroupPeriod::query()
            ->where('external_id', $this->pricingListLine->priceListPeriodId)
            ->first();
    }
}
