<?php

namespace Daylight\Connector\Exact\Listeners;

use App\Models\User;
use Daylight\Connector\Exact\Events\WebhookReceived;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateArticle;
use Daylight\Connector\Exact\Jobs\UpdateContact;
use Daylight\Connector\Exact\Jobs\UpdateCustomer;
use Daylight\Connector\Exact\Jobs\UpdateOrder;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ProcessWebhook implements ShouldQueue
{
    use SerializesModels;

    public function __construct(
        public Exact $connector
    ) {
        //
    }

    public function handle(WebhookReceived $event): void
    {
        $method = $this->getMethod($event->webhookData);

        if (method_exists($this, $method)) {
            $this->$method($this->getKey($event->webhookData));
        } else {
            \Log::channel('webhooks')->error('Method '.$method.' does not exist.');
        }
    }

    public function updateAccount(string $customerId): void
    {
        try {
            $customer = $this->connector
                ->customers()
                ->get(
                    mapper: $this->connector->getCustomerMapper(),
                    options: [
                        'id' => $customerId,
                    ])
                ->dtoOrFail();
        } catch (\Exception $e) {
            Log::channel('webhooks')->error('Failed to fetch customer with ID '.$customerId.': '.$e->getMessage());

            return;
        }

        UpdateCustomer::dispatch($customer)->onQueue('priority');
    }

    public function deleteAccount(string $customerId): void
    {
        daylightModel('customer')::query()
            ->where('external_id', $customerId)
            ->delete();
    }

    public function updateItem(string $articleId): void
    {
        try {
            $article = $this->connector
                ->products()
                ->get(
                    mapper: $this->connector->getProductMapper(),
                    id: $articleId,
                )
                ->dtoOrFail();
        } catch (\Exception $e) {
            Log::channel('webhooks')->error('Failed to fetch article with ID '.$articleId.': '.$e->getMessage());

            return;
        }

        UpdateArticle::dispatch($article)->onQueue('priority');
    }

    public function deleteItem(string $articleId): void
    {
        daylightModel('variant')::query()
            ->where('external_id', $articleId)
            ->delete();
    }

    public function updateContact(string $contactId): void
    {
        try {
            $contact = $this->connector
                ->contacts()
                ->get(
                    mapper: $this->connector->getContactMapper(),
                    id: $contactId,
                )
                ->dtoOrFail();
        } catch (\Exception $e) {
            Log::channel('webhooks')->error('Failed to fetch contact with ID '.$contactId.': '.$e->getMessage());

            return;
        }

        UpdateContact::dispatch($contact)->onQueue('priority');
    }

    public function deleteContact(string $contactId): void
    {
        User::query()
            ->where('external_id', $contactId)
            ->delete();
    }

    public function updateSalesOrder(string $orderId): void
    {
        try {
            $order = $this->connector
                ->orders()
                ->get(
                    mapper: $this->connector->getOrderMapper(),
                    options: [
                        'id' => $orderId,
                    ],
                )
                ->dtoOrFail();
        } catch (\Exception $e) {
            Log::channel('webhooks')->error('Failed to fetch contact with ID '.$orderId.': '.$e->getMessage());

            return;
        }

        UpdateOrder::dispatch($order)->onQueue('priority')
            ->delay(now()->addMinutes(5));
    }

    public function deleteSalesOrder(string $orderId): void
    {
        daylightModel('order')::query()
            ->where('external_id', $orderId)
            ->delete();
    }

    public function updateSalesOrderLine(string $salesOrderLineId): void
    {
        try {
            $orderLine = $this->connector
                ->orders()
                ->line(
                    mapper: $this->connector->getOrderItemMapper(),
                    id: $salesOrderLineId,
                )
                ->dtoOrFail();
        } catch (\Exception $e) {
            Log::channel('webhooks')->error('Failed to fetch sales order line with ID '.$salesOrderLineId.': '.$e->getMessage());

            return;
        }

        $this->updateSalesOrder($orderLine);
    }

    private function getMethod(array $requestData): string
    {
        return strtolower($requestData['Action']).ucfirst(Str::singular($requestData['Topic']));
    }

    private function getKey(array $requestData): ?string
    {
        return $requestData['Key'] ?? null;
    }
}
