<?php

namespace Daylight\Connector\Exact;

use Daylight\Connector\Connector;
use Daylight\Connector\Exact\Console\Commands\GetAccessToken;
use Daylight\Connector\Exact\Console\Commands\Install;
use Daylight\Connector\Exact\Console\Commands\SubscribeWebhooks;
use Daylight\Connector\Exact\Console\Commands\SyncDeletions;
use Daylight\Connector\Exact\Console\Commands\UpdateAddresses;
use Daylight\Connector\Exact\Console\Commands\UpdateArticleCharges;
use Daylight\Connector\Exact\Console\Commands\UpdateArticlePackaging;
use Daylight\Connector\Exact\Console\Commands\UpdateArticlePrices;
use Daylight\Connector\Exact\Console\Commands\UpdateArticles;
use Daylight\Connector\Exact\Console\Commands\UpdateContacts;
use Daylight\Connector\Exact\Console\Commands\UpdateCustomers;
use Daylight\Connector\Exact\Console\Commands\UpdateDocumentAttachments;
use Daylight\Connector\Exact\Console\Commands\UpdateDocuments;
use Daylight\Connector\Exact\Console\Commands\UpdateOrders;
use Daylight\Connector\Exact\Console\Commands\UpdatePricingListLines;
use Daylight\Connector\Exact\Console\Commands\UpdatePricingListPeriods;
use Daylight\Connector\Exact\Console\Commands\UpdatePricingLists;
use Daylight\Connector\Exact\Console\Commands\UpdateUnits;
use Daylight\Connector\Exact\Http\Controllers\OAuth2Controller;
use Daylight\Connector\Exact\Http\Controllers\WebhookController;
use Daylight\Connector\Exact\Models\Charge;
use Daylight\Connector\Exact\Models\CustomerGroupPeriod;
use Daylight\Connector\Exact\Models\Invoice;
use Daylight\Connector\Exact\Models\Unit;
use Daylight\Connector\Registries\ConnectorRegistry;
use Daylight\Core\Models\Customer;
use Daylight\Core\Models\CustomerGroup;
use Daylight\Core\Models\Price;
use Daylight\Core\Models\Variant;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;

class ExactServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->register(ExactEventServiceProvider::class);

        $this->commands([
            Install::class,
            GetAccessToken::class,
            UpdateUnits::class,
            UpdateAddresses::class,
            UpdateArticles::class,
            UpdateArticleCharges::class,
            UpdateArticlePackaging::class,
            UpdateArticlePrices::class,
            UpdatePricingLists::class,
            UpdatePricingListPeriods::class,
            UpdatePricingListLines::class,
            UpdateOrders::class,
            UpdateCustomers::class,
            UpdateContacts::class,
            UpdateDocuments::class,
            UpdateDocumentAttachments::class,
            SyncDeletions::class,
            SubscribeWebhooks::class,
        ]);

        $this->loadMigrationsFrom(__DIR__.'/../database/migrations');

        $this->publishes([
            __DIR__.'/../config/exact.php' => config_path('exact.php'),
        ]);

        $this->mergeConfigFrom(__DIR__.'/../config/exact.php', 'exact');

        Route::macro('exact', function () {
            Route::prefix('exact')->group(function () {
                Route::get('/auth/redirect', [OAuth2Controller::class, 'redirectToProvider'])
                    ->name('exact.auth.redirect');

                Route::get('/auth/callback', [OAuth2Controller::class, 'callback'])
                    ->name('exact.auth.callback');

                Route::post('/webhook', WebhookController::class)
                    ->name('exact.webhook');
            });
        });

        daylightModel('variant')::resolveRelationUsing('unit', function (Variant $variant) {
            return $variant->belongsTo(Unit::class, 'unit_id');
        });

        daylightModel('variant')::resolveRelationUsing('basePrices', function (Variant $variant) {
            return $variant->prices()->whereMinQuantity(1)->whereNull('customer_group_id')->whereNull('customer_id');
        });

        daylightModel('variant')::resolveRelationUsing('charges', function (Variant $variant) {
            return $variant->hasMany(Charge::class, 'variant_id', 'id');
        });

        daylightModel('price')::resolveRelationUsing('customer', function (Price $price) {
            return $price->belongsTo(daylightModel('customer'), 'customer_id');
        });

        daylightModel('customer')::resolveRelationUsing('invoices', function (Customer $customer) {
            return $customer->hasMany(Invoice::class, 'customer_id', 'id');
        });

        daylightModel('customer_group')::resolveRelationUsing('periods', function (CustomerGroup $customerGroup) {
            return $customerGroup->hasMany(CustomerGroupPeriod::class, 'customer_group_id', 'id');
        });
    }

    public function boot(): void
    {
        ConnectorRegistry::addConnector(
            Connector::create()
                ->setKey('exact')
                ->setConnector(Exact::class)
        );
    }
}
