<?php

namespace Daylight\Connector\Exact\Listeners;

use Daylight\Connector\Exact\Exact;
use Daylight\Core\Events\OrderCreated;
use Daylight\Core\Models\Enums\ShippingMethod;
use Daylight\Core\Models\Order;
use Daylight\Core\Models\OrderItem;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\SerializesModels;

class CreateOrderOnConnector implements ShouldQueue
{
    use SerializesModels;

    public function __construct(
        protected Exact $exact
    ) {
        //
    }

    public function handle(OrderCreated $event): void
    {
        $orderData = $this->getOrderData($event->getOrder());

        $response = $this->exact->orders()->create(
            mapper: $this->exact->getOrderMapper(),
            data: $orderData,
        );

        if ($response->successful()) {
            $createdOrder = $response->dto();

            $event->getOrder()->update([
                'external_id' => $createdOrder->id,
            ]);

            $exactOrder = $this->exact->orders()->get(
                mapper: $this->exact->getOrderMapper(),
                options: [
                    'id' => $createdOrder->id,
                ],
            )->dtoOrFail();

            foreach ($event->getOrder()->items as $item) {
                $exactItem = $exactOrder->orderLines->firstWhere('sku', $item->sku);

                if ($exactItem) {
                    $item->update([
                        'external_id' => $exactItem->id,
                    ]);
                }
            }

            $event->getOrder()->updates()->create([
                'type' => 'system',
                'content' => __('Order created on Exact with ID :id', ['id' => $createdOrder->id]),
            ]);
        } else {
            \Log::error('Failed to create order on Exact', [
                'order_id' => $event->getOrder()->id,
                'response_status' => $response->status(),
                'response_body' => $response->body(),
            ]);
        }
    }

    private function getOrderData(Order $order): array
    {
        return [
            'OrderedBy' => $order->customer->external_id,
            'OrderedByContactPerson' => $order->user->external_id ?? null,
            'OrderNumber' => $order->number ?? null,
            'DeliveryAddress' => $order->shippingAddress->external_id ?? null,
            'PaymentCondition' => $order->payment_method === 'invoice'
                ? $order->customer->payment_condition_code
                : '0',
            'PaymentReference' => $order->payment_gateway_id ?? null,
            'YourRef' => $order->reference ?? null,
            'ShippingMethod' => $this->getShippingMethodId($order),
            'Description' => $order->notes,
            'WarehouseID' => '28612bbb-a7e5-4686-a4d1-6f15c4249a7f', // Raamsdonkveer
            'SalesOrderLines' => [
                ...$order->items->map(function (OrderItem $item) {
                    return [
                        'Item' => $item->external_id,
                        'Description' => $item->name,
                        'Quantity' => $item->quantity,
                        'UnitPrice' => $item->price,
                    ];
                })->toArray(),
                ...$this->getShippingLineData($order),
            ],
        ];
    }

    private function getShippingMethodId(Order $order): string
    {
        if ($order->shipping_method === ShippingMethod::PICKUP) {
            return '8ac30eac-6da5-4bba-aee4-0987cb031621'; // Pick-up
        }

        if ($order->customer->package_delivery) {
            return '69ae9bcf-657b-4a3d-aaa9-1b90c13198b4'; // Package delivery
        }

        return '7e83c1bf-cdd0-409b-a9aa-b705d8073c9f'; // By us
    }

    private function getShippingLineData(Order $order): array
    {
        return [
            [
                'Item' => 'cdf0ad77-312c-4a0e-8952-aa329fbd6226',
                'Description' => 'Verzendkosten',
                'Quantity' => 1,
                'UnitPrice' => $order->shipping,
            ],
        ];
    }
}
