<?php

namespace Daylight\Connector\Exact\Console\Commands;

use Daylight\Connector\Exact\Entities\PaginatedResource;
use Daylight\Connector\Exact\Entities\ProductPrice;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateArticlePrice;
use Daylight\Connector\Exact\Models\Import;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\LazyCollection;

class UpdateArticlePrices extends Command
{
    protected $signature = 'exact:update-article-prices {--debug : Show debug information} {--force : Force update even if no changes}';

    protected $description = 'Update article prices from Exact Online';

    protected int $totalRows = 0;

    public function handle(): int
    {
        $this->info('Updating article prices...');

        $skipToken = null;
        $currentCursor = $this->option('force') ? 1 : $this->getCurrentCursor();
        $cursor = max($currentCursor, 0);

        $this->logMemoryUsage($this->option('debug'));

        $this->line('Fetching items starting from cursor: '.$currentCursor);

        LazyCollection::make(function () use (&$skipToken, &$currentCursor) {
            do {
                try {
                    $results = $this->getItems($currentCursor, $skipToken);
                } catch (\Exception $e) {
                    $results = null;
                }

                if (is_null($results) || $results->getData()->isEmpty()) {
                    return;
                }

                $skipToken = $results->getSkipToken();

                yield $results;

            } while ($results->hasMore());
        })->each(function (PaginatedResource $items, $index) use (&$cursor) {
            sleep(2);

            $this->info('Processing page '.($index + 1).' with '.$items->getData()->count().' items.');

            $this->totalRows += $items->getData()->count();

            $jobs = $items->getData()->map(function (ProductPrice $productPrice) use (&$cursor) {
                $cursor = max($cursor, $productPrice->timestamp);

                return new UpdateArticlePrice(
                    price: $productPrice,
                );
            });

            Bus::batch($jobs)
                ->name('exact:product-prices:iteration:'.($index + 1))
                ->onQueue('exact')
                ->dispatch();

            unset($jobs);
            unset($items);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page '.($index + 1));

            $this->logMemoryUsage($this->option('debug'));
        });

        Import::updateOrCreate([
            'type' => 'product-prices',
        ], [
            'cursor' => $cursor,
        ]);

        $this->logMemoryUsage($this->option('debug'));

        $this->info('Total items processed: '.$this->totalRows);

        return Command::SUCCESS;
    }

    protected function getCurrentCursor(): int
    {
        return Import::query()
            ->firstWhere('type', 'product-prices')
            ?->cursor ?? 0;
    }

    protected function getItems(int $cursor, ?string $skipToken = null): ?PaginatedResource
    {
        return $this->connector()
            ->products()
            ->prices(
                mapper: $this->connector()->getProductPriceMapper(),
                options: [
                    'cursor' => $cursor,
                    'skiptoken' => $skipToken,
                ]
            )
            ->dto();
    }

    private function connector(): Exact
    {
        return app(Exact::class);
    }

    private function logMemoryUsage(bool $debug = false): void
    {
        if (! $debug) {
            return;
        }

        $this->line('--> Memory Usage: '.round(memory_get_usage() / 1024 / 1024, 2).' MB');
        $this->line('--> Peak Memory Usage: '.round(memory_get_peak_usage() / 1024 / 1024, 2).' MB');
        $this->newLine();
    }
}
