<?php

namespace Daylight\Connector\Exact\Console\Commands;

use Daylight\Connector\Exact\Entities\PaginatedResource;
use Daylight\Connector\Exact\Entities\ProductCharge;
use Daylight\Connector\Exact\Exact;
use Daylight\Connector\Exact\Jobs\UpdateProductCharge;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\LazyCollection;

class UpdateArticleCharges extends Command
{
    protected $signature = 'exact:update-article-charges {--debug : Show debug information}';

    protected $description = 'Update article charges from Exact Online';

    protected int $totalRows = 0;

    public function handle(): int
    {
        $this->info('Updating article charges...');

        $skipToken = null;

        $this->logMemoryUsage($this->option('debug'));

        LazyCollection::make(function () use (&$skipToken) {
            do {
                $this->line('Fetching items with skip token: '.($skipToken ?? 'none'));

                try {
                    $results = $this->getItems($skipToken);
                } catch (\Exception $e) {
                    $results = null;
                }

                if (is_null($results) || $results->getData()->isEmpty()) {
                    return;
                }

                $skipToken = $results->getSkipToken();

                yield $results;

            } while ($results->hasMore());
        })->each(function (PaginatedResource $items, $index) {
            sleep(2);

            $this->info('Processing page '.($index + 1).' with '.$items->getData()->count().' items.');

            $this->totalRows += $items->getData()->count();

            $jobs = $items->getData()->map(function (ProductCharge $item) {
                return new UpdateProductCharge(
                    productCharge: $item
                );
            });

            Bus::batch($jobs)
                ->name('exact:product-charges:iteration:'.($index + 1))
                ->onQueue('exact')
                ->dispatch();

            unset($jobs);
            unset($items);

            gc_collect_cycles();

            $this->line('-> Dispatched jobs for page '.($index + 1));

            $this->logMemoryUsage($this->option('debug'));
        });

        $this->logMemoryUsage($this->option('debug'));

        $this->info('Total items processed: '.$this->totalRows);

        return Command::SUCCESS;
    }

    protected function getItems(?string $skipToken = null): ?PaginatedResource
    {
        return $this->connector()
            ->products()
            ->charges(
                mapper: $this->connector()->getProductChargeMapper(),
                options: [
                    'skiptoken' => $skipToken,
                ]
            )
            ->dto();
    }

    private function connector(): Exact
    {
        return app(Exact::class);
    }

    private function logMemoryUsage(bool $debug = false): void
    {
        if (! $debug) {
            return;
        }

        $this->line('--> Memory Usage: '.round(memory_get_usage() / 1024 / 1024, 2).' MB');
        $this->line('--> Peak Memory Usage: '.round(memory_get_peak_usage() / 1024 / 1024, 2).' MB');
        $this->newLine();
    }
}
