<?php

namespace Daylight\Connector\Exact;

use Daylight\Connector\Contracts\MapperInterface;
use Daylight\Connector\Exact\Mappers\AddressMapper;
use Daylight\Connector\Exact\Mappers\ContactMapper;
use Daylight\Connector\Exact\Mappers\CustomerMapper;
use Daylight\Connector\Exact\Mappers\DeletionMapper;
use Daylight\Connector\Exact\Mappers\DocumentAttachmentMapper;
use Daylight\Connector\Exact\Mappers\DocumentMapper;
use Daylight\Connector\Exact\Mappers\OrderItemMapper;
use Daylight\Connector\Exact\Mappers\OrderMapper;
use Daylight\Connector\Exact\Mappers\PricingListLinesMapper;
use Daylight\Connector\Exact\Mappers\PricingListMapper;
use Daylight\Connector\Exact\Mappers\PricingListPeriodMapper;
use Daylight\Connector\Exact\Mappers\ProductChargeMapper;
use Daylight\Connector\Exact\Mappers\ProductExtraFieldMapper;
use Daylight\Connector\Exact\Mappers\ProductMapper;
use Daylight\Connector\Exact\Mappers\ProductPriceMapper;
use Daylight\Connector\Exact\Mappers\UnitMapper;
use Daylight\Connector\Exact\Middleware\AuthenticateWithToken;
use Daylight\Connector\Exact\Resources\Addresses;
use Daylight\Connector\Exact\Resources\Contacts;
use Daylight\Connector\Exact\Resources\Deletions;
use Daylight\Connector\Exact\Resources\Documents;
use Daylight\Connector\Exact\Resources\Orders;
use Daylight\Connector\Exact\Resources\PricingLists;
use Daylight\Connector\Exact\Resources\Products;
use Daylight\Connector\Exact\Resources\Units;
use Daylight\Connector\Exact\Resources\Webhooks;
use Daylight\Connector\Exceptions\RequestNotFoundException;
use Daylight\Connector\Services\BaseConnector;
use Illuminate\Support\Str;
use Saloon\Helpers\OAuth2\OAuthConfig;
use Saloon\Http\PendingRequest;
use Saloon\Traits\OAuth2\AuthorizationCodeGrant;

class Exact extends BaseConnector
{
    use AuthorizationCodeGrant;

    protected string $name = 'Exact Online';

    protected string $namespace = 'Daylight\Connector\Exact';

    public function getName(): string
    {
        return $this->name;
    }

    public function getKey(): string
    {
        return Str::slug($this->name);
    }

    public function resolveBaseUrl(): string
    {
        return 'https://start.exactonline.nl';
    }

    public function getTokenStorage(): TokenStorage
    {
        return app(TokenStorage::class);
    }

    public function boot(PendingRequest $pendingRequest): void
    {
        $pendingRequest->middleware()->onRequest(
            new AuthenticateWithToken(
                tokenStorage: $this->getTokenStorage()
            )
        );
    }

    protected function defaultOauthConfig(): OAuthConfig
    {
        return OAuthConfig::make()
            ->setClientId(config('services.exact.client_id'))
            ->setClientSecret(config('services.exact.client_secret'))
            ->setRedirectUri(route('exact.auth.callback'))
            ->setAuthorizeEndpoint('https://start.exactonline.nl/api/oauth2/auth')
            ->setTokenEndpoint('https://start.exactonline.nl/api/oauth2/token')
            ->setUserEndpoint('https://start.exactonline.nl/api/v1/current/Me');
    }

    public function defaultHeaders(): array
    {
        return [
            'Accept' => 'application/json',
        ];
    }

    public function products(): Products
    {
        return new Products($this);
    }

    public function units(): Units
    {
        return new Units($this);
    }

    public function pricingLists(): PricingLists
    {
        return new PricingLists($this);
    }

    public function deletions(): Deletions
    {
        return new Deletions($this);
    }

    public function contacts(): Contacts
    {
        return new Contacts($this);
    }

    public function addresses(): Addresses
    {
        return new Addresses($this);
    }

    public function orders(): Orders
    {
        return new Orders($this);
    }

    public function documents(): Documents
    {
        return new Documents($this);
    }

    public function webhooks(): Webhooks
    {
        return new Webhooks($this);
    }

    public function getAddressMapper(): MapperInterface
    {
        return new AddressMapper;
    }

    public function getProductMapper(): MapperInterface
    {
        return new ProductMapper;
    }

    public function getProductPriceMapper(): MapperInterface
    {
        return new ProductPriceMapper;
    }

    public function getOrderMapper(): MapperInterface
    {
        return new OrderMapper;
    }

    public function getOrderItemMapper(): MapperInterface
    {
        return new OrderItemMapper;
    }

    public function getUnitMapper(): MapperInterface
    {
        return new UnitMapper;
    }

    public function getPricingListMapper(): MapperInterface
    {
        return new PricingListMapper;
    }

    public function getPricingListPeriodMapper(): MapperInterface
    {
        return new PricingListPeriodMapper;
    }

    public function getPricingListLinesMapper(): MapperInterface
    {
        return new PricingListLinesMapper;
    }

    public function getDeletionMapper(): MapperInterface
    {
        return new DeletionMapper;
    }

    public function getProductChargeMapper(): MapperInterface
    {
        return new ProductChargeMapper;
    }

    public function getProductExtraFieldMapper(): MapperInterface
    {
        return new ProductExtraFieldMapper;
    }

    public function getCustomerMapper(): MapperInterface
    {
        return new CustomerMapper;
    }

    public function getContactMapper(): MapperInterface
    {
        return new ContactMapper;
    }

    public function getDocumentMapper(): MapperInterface
    {
        return new DocumentMapper;
    }

    public function getDocumentAttachmentMapper(): MapperInterface
    {
        return new DocumentAttachmentMapper;
    }

    public function requestHandler(string $request, MapperInterface $mapper, array $options = [])
    {
        if (! $request) {
            throw RequestNotFoundException::noRequestSet();
        }

        $class = $this->namespace.'\\Requests\\'.$request;

        if (! class_exists($class)) {
            throw RequestNotFoundException::forRequest($request);
        }

        return $this->send(
            new $class($mapper, $options)
        );
    }
}
